import java.io.*;
import java.net.*;
import java.util.*;

// ReflectorListener thread listens for packets
// and notifies one or more interested threads
// who register as PacketListenerInterfaces.

public class ReflectorListener extends Thread {
  private InetAddress listenAddr;
  private int         listenPort;
  private int         mode;

  private Vector      packetListeners;
  private Logger      logger;

  private static final int MAX_PACKET_SIZE = 1500;

  public ReflectorListener(Address a, int mode, Logger logger) {
    listenAddr  = a.getAddress();
    listenPort  = a.getPort();
    this.mode   = mode;
    this.logger = logger;

    packetListeners = new Vector();
  }

  public void run() {
    // create a unicast or multicast socket
    // depending on the mode and listen for packets.

    switch (mode) {
    case Reflector.MODE_UNI_TO_MULTI:
      DatagramSocket ds = initUnicastSocket();
      if (ds != null) listen(ds);
      break;
    case Reflector.MODE_MULTI_TO_UNI:
      MulticastSocket mc = initMulticastSocket();
      if (mc != null) listen((DatagramSocket)mc);
      break;
    default:
      break;
    }
  }

  private MulticastSocket initMulticastSocket() {
    // initialize a MulticastSocket and join the group

    MulticastSocket mc;
    try {
      mc = new MulticastSocket(listenPort);
      mc.joinGroup(listenAddr);
    } catch (Exception e) {
      System.err.println("Failed to create MulticastSocket on " + 
                         "port " + listenPort);
      return(null);
    }
    return(mc);
  }

  private DatagramSocket initUnicastSocket() {
    // initialize a DatagramSocket

    DatagramSocket ds;
    try {
      ds = new DatagramSocket(listenPort);
    } catch (Exception e) {
      System.err.println("Failed to create DatagramSocket on " 
                         + "port " + listenPort);
      return(null);
    }
    return(ds);
  }
  
  private void listen(DatagramSocket ds) {
    // loop forever listening to packets, when they
    // arrive log them and notify all interested threads.

    byte[] buffer;
    DatagramPacket packet;
    
    while (true) {
      try {
        buffer = new byte[MAX_PACKET_SIZE];
        packet = new DatagramPacket(buffer, buffer.length);

        ds.receive(packet);
    
        logger.log("Packet received, " + packet.getLength() 
                   + " bytes");
        eventNotify(packet); 

      } catch (IOException e) {
        System.err.println("Error receiving packet\n");
        e.printStackTrace();
      }
    }
  }

  public void addPacketListener(PacketListenerInterface pl) {
     // add interested thread to listeners vector
     packetListeners.addElement(pl);
  }

  public void removePacketListener(PacketListenerInterface pl) {
     // remove thread to listeners vector
     packetListeners.removeElement(pl);
  }

  private void eventNotify(DatagramPacket packet) {
    // notify all registered threads that a packet has arrived
    // using the packetReceived(DatagramPacket) method.

    for (Enumeration e = packetListeners.elements(); 
         e.hasMoreElements(); ) {

      PacketListenerInterface pl = 
                    (PacketListenerInterface) e.nextElement();
      pl.packetReceived(packet);
    } 
  }
}
