/****************************************************************************
* ScanRtpFile.java
*
* PURPOSE
* Scan a file created by rtpdump from the end backwards to determine
* the duration of the stream.
*
* MCONTROL
* Written by David Makofske and Kevin Almeroth
* University of California at Santa Barbara
* Department of Computer Science
* Neworking and Multimedia Systems Laboratory
* http://imj.ucsb.edu/mcontrol
*
* COPYRIGHT
* Copyright 1998
* University of California at Santa Barbara
* Networking and Multimedia Systems Lab
* Santa Barbara, CA  93106
* ALL RIGHTS RESERVED
*
* This material may be modified, copied and redistributed, both within
* the recipient's organization and externally, subject to the following
* restrictions:  (a)  The recipient may not derive income from the
* University of California at Santa Barbara (herein "UCSB") information
* itself;  (b)  In any material based on this information, the recipient
* agrees to acknowledge UCSB; (c)  Any copies made of this material must be
* accompanied by the following copyright notice: "Copyright 1998 University
* of California at Santa Barbara. Santa Barbara, CA 93106. All Rights
* Reserved."; and (d)  The recipient agrees to obey all U.S. Government
* restrictions governing redistribution or export of such information.
* These restrictions may apply to redistribution within an international
* organization. UCSB makes no warranties or representations, either
* expressed or implied, with respect to the results containied herein, its
* quality, merchantability, performance or fitness for a particular purpose.
* In no event shall UCSB or its developers, directors, officers, employees
* or affiliates be liable for direct, incidental, indirect, special or
* consequential damages (including damages or loss of business profits,
* business interruption, loss of business information and the like)
* resulting from any defect in this material or its documentation or
* arising out of the use or inability to use this material or accompanying
* documentation even if UCSB, an authorized representative or a UCSB
* affiliate has been advised of the possibility of such damage.  UCSB
* makes no representation or warranty regarding the results obtainable
* through use of this material.  No oral or written information or advice
* given by UCSB, its dealers, distributors, agents, affiliates, developers,
* directors, officers or employees shall create a warranty or in any way
* increase the scope of this warranty.
****************************************************************************/

import java.io.*;

public class ScanRtpFile {

  /* ScanRtpFile:
     scans for the last rtpdump packet header in the file to find the
     time in milliseconds since the beginning of the file. Returns 0
     on error or if duration cannot be computed. */

  private RandomAccessFile raf;
  private long filelen=0;        // total size of rtpdump file in bytes.
  private long file_offset=0;    // bytes from end of file to be examined.

  private String HDR_MAGIC = "#!rtpplay1.0";

  private long HDR_length=0;     // packet length including this header.
  private int  HDR_plen=0;       // packet length not including this header
                                 // or zerofor RTCP packets.
  private long HDR_offset=0;     // offset since start of dump in seconds.

  private int HDR_SIZE=8;        // size of rtpdump header, diff between
                                 // HDR_length and HDR_plen.

  private int MAX_TRIES=500;

  public ScanRtpFile() { }

  public long getDuration(File file) {

    if (! file.exists()) {
      System.out.println("Cannot find file " + file.getName() + ".");
      return(0);
    }

    try {
      raf = new RandomAccessFile(file,"r");
    } catch (IOException e) {
      System.out.println("Error reading file " + file.getName() + ".");
      System.out.println("Error thrown was " + e.toString());
      return(0);
    }
    
    try {
      filelen = raf.length();
    } catch (IOException ioe) {
      System.out.println("Error getting " + file.getName() + " file length.");
      System.out.println("Error thrown was " + ioe.toString());
      return(0);
    }

    if (filelen == 0) {
      System.out.println("File was size 0!");
      return(0);
    }

    if (! ValidHeader(raf)) 
      return(0);

//    System.out.println("DEBUG: File size is " + filelen);

    file_offset=HDR_SIZE;

    long len1, len2;

    while (true) {
      len1 = findLen();
      len2 = findLen();
      if (((len1 - len2) < 2000) && 
          (len1 != 0) && (len2 != 0)) {
        // our method of finding the length is not very safe, so try
        // it twice and compare the results. If they are within 2000 millisecs
        // (2 seconds), they are probably OK, otherwise keep looking.

//        System.out.println("DEBUG: lengths were " + len1 + " and " + len2);
        break;
      }
    }
   
    return(len1);
  }

  boolean ValidHeader(RandomAccessFile raf) {
    String line;

    try {
      line = raf.readLine();
    } catch (IOException ioe) {
      System.out.println("Error reading rtpdump version header.");
      System.out.println("Error thrown was " + ioe.toString());
      return(false);
    }

//    System.out.println("DEBUG: " + line);

    line = line.substring(0, HDR_MAGIC.length());

    if (! line.equals(HDR_MAGIC)) {
      System.out.println("Invalid rtpdump file.");
      System.out.println("Expected \"" + HDR_MAGIC + "\", " +
                         " got \"" + line + "\"");
      return(false);
    }
    
    return(true);
  }
     
  private long findLen() {
    int count=0;

    boolean found=false;

    while (!found) {

/*** count++;
     if (count > MAX_TRIES) break; ***/

      try {

//        System.out.println("DEBUG: seeking to " + (filelen - file_offset));
        raf.seek(filelen - file_offset);

        HDR_length = raf.readUnsignedShort();

//        HDR_plen = raf.readUnsignedShort();

// because of a bug in rtpdump, the HDR_plen field could be in either
// big endian or little endian byte order. As a result, we have to check
// both:
        int low  = raf.readByte() & 0x00ff;
        int high = raf.readByte() & 0x00ff;
        HDR_plen = (low << 8 | high);
                        
//        System.out.println("DEBUG: file_offset = "+ file_offset + ", " +
//                           "HDR_length = " + HDR_length + ", " + 
//                           "HDR_plen = " + HDR_plen);

        if ((HDR_length - HDR_SIZE) == HDR_plen) {
//           System.out.println("DEBUG: Match Found!");
           found = true;
           break;
        }

        // now reverse the byte order:
        HDR_plen = (high << 8 | low);

//        System.out.println("DEBUG: little endian HDR_plen = " + HDR_plen);

        if ((HDR_length - HDR_SIZE) == HDR_plen) {
//           System.out.println("DEBUG: Match Found!");
           found = true;
           break;
        }

      } catch (IOException ioe) {
        System.out.println("Error reading rtpdump file.");
        System.out.println("Exception thrown was " + ioe.toString());
      }

      file_offset += 1;
      if (file_offset >= filelen) {
        // we have scanned the entire file without a successfull match
        break;
      }
    }

    // if we have successfully identified a header, parse the time offset
    // field from it:

    if (found) {
      byte b[] = new byte[4];
      try {
        b[0] = raf.readByte();
        b[1] = raf.readByte();
        b[2] = raf.readByte();
        b[3] = raf.readByte();
      } catch (IOException ioe) {
        System.out.println("Error reading rtpdump file.");
        System.out.println("Exception thrown was " + ioe.toString());
      }

      int MASK = 0x00000000000000ff;
      HDR_offset = ((b[0] & MASK) << 24) | ((b[1] & MASK) << 16) |
                   ((b[2] & MASK) << 8) | (b[3] & MASK);

      file_offset+=1;

      if (HDR_offset > 86400000) {
        // Additional sanity check. If offset is > 86400000 then the recording
        // is longer than 24 hours, which is unlikely. Kill it as erroneous.
        HDR_offset = 0;
      } 
   }

//    System.out.println("DEBUG: HDR_offset = " + HDR_offset);

    return(HDR_offset);
  }
}

