/****************************************************************************
* VCRButton.java
*
* PURPOSE
* Custom component for the VCR buttons.
*
* MCONTROL
* Written by David Makofske and Kevin Almeroth
* University of California at Santa Barbara
* Department of Computer Science
* Neworking and Multimedia Systems Laboratory
* http://imj.ucsb.edu/mcontrol
*
* COPYRIGHT
* Copyright 1998
* University of California at Santa Barbara
* Networking and Multimedia Systems Lab
* Santa Barbara, CA  93106
* ALL RIGHTS RESERVED
*
* This material may be modified, copied and redistributed, both within
* the recipient's organization and externally, subject to the following
* restrictions:  (a)  The recipient may not derive income from the
* University of California at Santa Barbara (herein "UCSB") information
* itself;  (b)  In any material based on this information, the recipient
* agrees to acknowledge UCSB; (c)  Any copies made of this material must be
* accompanied by the following copyright notice: "Copyright 1998 University
* of California at Santa Barbara. Santa Barbara, CA 93106. All Rights
* Reserved."; and (d)  The recipient agrees to obey all U.S. Government
* restrictions governing redistribution or export of such information.
* These restrictions may apply to redistribution within an international
* organization. UCSB makes no warranties or representations, either
* expressed or implied, with respect to the results containied herein, its
* quality, merchantability, performance or fitness for a particular purpose.
* In no event shall UCSB or its developers, directors, officers, employees
* or affiliates be liable for direct, incidental, indirect, special or
* consequential damages (including damages or loss of business profits,
* business interruption, loss of business information and the like)
* resulting from any defect in this material or its documentation or
* arising out of the use or inability to use this material or accompanying
* documentation even if UCSB, an authorized representative or a UCSB
* affiliate has been advised of the possibility of such damage.  UCSB
* makes no representation or warranty regarding the results obtainable
* through use of this material.  No oral or written information or advice
* given by UCSB, its dealers, distributors, agents, affiliates, developers,
* directors, officers or employees shall create a warranty or in any way
* increase the scope of this warranty.
****************************************************************************/

import java.awt.*;
import java.awt.event.*;
import java.io.*;

class VCRButton extends Component {

  public static final int BORDER_XWIDTH  = 5;
  public static final int BORDER_YHEIGHT = 5;

  private int width, height;
  private String imagePath;
  private Image pressedImage, unpressedImage;
  private boolean pressed = false;
  private String altText;
  private boolean on = false;
  private Color mygreen;
  private boolean imageFound = false;
  private ActionListener actionListener;
  private String pressedSrc, unpressedSrc;

  public VCRButton(String imagePath, String pressedSrc, String unpressedSrc, 
                     boolean on, String altText) {
    this.imagePath = imagePath;
    this.pressedSrc = pressedSrc;
    this.unpressedSrc = unpressedSrc;
    this.on = on;
    this.altText = altText;
     
    loadImages();
                                                                         
    if (! imageFound) {
      width=43 + 2 * BORDER_XWIDTH;
      height=43 + 2 * BORDER_YHEIGHT;
      setSize(width,height);
      mygreen = new Color((float)0.2, (float)0.6, (float)0.2);
    }

    enableEvents(AWTEvent.MOUSE_EVENT_MASK);

  }

  public void loadImages() {
    MediaTracker tracker;

    File f = new File(imagePath, pressedSrc);
    if (!f.exists()) {
      imageFound=false;
      return;
    }
    pressedSrc = imagePath + System.getProperty("file.separator") + pressedSrc;
    pressedImage = Toolkit.getDefaultToolkit().getImage(pressedSrc);
    tracker = new MediaTracker(this);
    tracker.addImage(pressedImage,0);
    try { tracker.waitForID(0); }
    catch (InterruptedException e) {}

    f = new File(imagePath,unpressedSrc);
    if (!f.exists()) {
      imageFound=false;
      return;
    }
    unpressedSrc = imagePath + System.getProperty("file.separator") 
                   + unpressedSrc;
    unpressedImage = Toolkit.getDefaultToolkit().getImage(unpressedSrc);
    tracker = new MediaTracker(this);
    tracker.addImage(unpressedImage,0);
    try { tracker.waitForID(0); }
    catch (InterruptedException e) {}
  
    imageFound=true;

    width = pressedImage.getWidth(this) + 2 * BORDER_XWIDTH;
    height = pressedImage.getHeight(this) + 2 * BORDER_YHEIGHT;
  }

  public void paint(Graphics g) {
    if (imageFound) {
      g.drawImage(on ? pressedImage : unpressedImage, 
                  BORDER_XWIDTH, BORDER_YHEIGHT, this);
    } else {
      if (on) 
        g.setColor(mygreen); 
      else
        g.setColor(Color.black);
      g.setFont(new Font("TimesRoman",Font.BOLD,20));
      FontMetrics fm = g.getFontMetrics(getFont());
      int x = ((width - (BORDER_XWIDTH*2) - fm.stringWidth(altText)) / 2);
      int y = (height / 2) + BORDER_YHEIGHT;
      g.drawString(altText,x,y);
    }

    g.setColor(Color.gray);
    for (int i=0; i <= (BORDER_XWIDTH+BORDER_YHEIGHT)/4; i++) {
      g.draw3DRect(i, i, width-2*i-1, height-2*i-1, !pressed);
    }
  }

  public Dimension getPreferredSize() {
    return getMinimumSize();
  }

  public Dimension getMinimumSize() {
    return new Dimension(width,height);
  }

  public void addActionListener(ActionListener listener) {
    actionListener = AWTEventMulticaster.add(actionListener, listener);
    enableEvents(AWTEvent.MOUSE_EVENT_MASK);
  }


  public void removeActionListener(ActionListener listener) {
    actionListener = AWTEventMulticaster.remove(actionListener, listener);
  }

  public void processMouseEvent(MouseEvent e) {
    if (e.getID() == MouseEvent.MOUSE_PRESSED) {
      pressed = true;
      repaint();
    } else if (e.getID() == MouseEvent.MOUSE_RELEASED) {
      if (actionListener != null) 
        actionListener.actionPerformed(new ActionEvent(this,
           ActionEvent.ACTION_PERFORMED, "" + this));
      if (pressed) {
        pressed = false;
        repaint();
      }
    }
    else if (e.getID() == MouseEvent.MOUSE_EXITED) {
      if (pressed) {
        pressed = false;
        repaint();
      }
    }
    super.processMouseEvent(e);
  }

  public void setOn(boolean on) {
    this.on = on;
    repaint();
  }
}
