#
#     File : parse.pl
#   Author : Robert Chalmers
#
# Original : November 22, 1999
#  Revised :
#
#  Content : main script for processing mtrace and rtcp logs
#

package main;

# import external classes/modules
use FileHandle;

# import internal classes/modules
use Mwalk::Args;
use Mwalk::Receiver;
use Mwalk::RTCP;
use Mwalk::MTrace;
use Mwalk::TraceSet;


# enforce strictness
use strict qw( vars refs subs );

# command-line arguments used to control execution
my $args = new Mwalk::Args;
# stats kept for each type of data (rtcp, mtrace, interval)
my %stats = ( counts => { min => undef, max => undef, with => undef, tot => undef, skip => undef },
              skips  => {none => 0, form => 0, cmd => 0, hopless => 0, start => 0, strict => 0, loop => 0}
            );
# list of processed receivers hashed by ip address
my %receivers;


# procedure declarations
use subs qw(initStats printUsage parseArgs parseRTCP parseMTrace saveStats saveData goInteractive);


# prepare the stats array
initStats;
# parse command-line options
parseArgs;

# mark the start time for processing
$args->setStart( time );

# parse rtcp file if provided
parseRTCP if $args->{rtcpFile};
# parse mtrace file if provided
parseMTrace if $args->{mtraceFile};

# mark the stop time for processing
$args->setStop( time );

# output if indicated
saveData;
saveStats if $args->{statFile};

# provide an interactive command line if requested
goInteractive if $args->{interactive};

exit;



#
# initilialize the statistics accumulators
#
sub initStats {

  # initialize stats for interval, rtcp, and mtrace data
  foreach my $stat (keys %{$stats{counts}}) {
    $stats{counts}{$stat} = { interval => 0, rtcp => 0, mtrace => 0 };
  }
  # set minimums arbitrarily high
  $stats{counts}{min}{interval} = $stats{counts}{min}{rtcp} = $stats{counts}{min}{mtrace} = 100000000;
}


#
# print script usage
#
# params
#   optional exit code
#
sub printUsage {

  my $code = shift;
  # default exit code to -1
  $code = -1 unless defined( $code );

  print <<EOU;
Usage: mwalk parse [-i|--interactive] [-v|--verbose] 
                   [-r|--rtcp=<rfile>] [-or|--output_rtcp]
                   [-m|--mtrace=<mfile>] [-om|--output_mtrace]
                   [-s|--stats=<sfile>] [-d|--dir=<outdir>] 
                   [-t|--timeout=<timeout>] [-f|--fussy]
                   [-a|--append] [-b|--build] [-h|--help]
Options:
  -i  --interactive    run in an interactive mode
  -v  --verbose        print verbose messages
  -l  --log            generate processing logs
  -r  --rtcp           source rtcp log to parse
  -or --output_rtcp    output parsed rtcp files
  -m  --mtrace         source mtrace log to parse
  -om --output_mtrace  output parsed mtrace files
  -s  --stats          generate statistics log [stats]
  -d  --dir            output directory [.]
  -t  --timeout        seconds to distinguish between rtcp intervals [240s]
  -f  --fussy          use strict parsing
  -a  --append         append to existing log files
  -b  --build          accumulate traces in memory
  -h  --help           print this message

EOU

  # exit with provided code
  exit $code;
}

   
#
# parse the command line options
#
sub parseArgs {

  # process the command line options
  $args->parse( "interactive!", "verbose!",
		"append!", "build!", "fussy!", "help", 
		"rtcp=s", "output_rtcp|or!",
		"mtrace=s", "output_mtrace|om!",
		"stats=s", "dir=s", "timeout=i" );

  # sanity check
  $args->{interactive} or $args->{rtcpFile} or $args->{mtraceFile}
                       or print "\nMust specify at least one input file (-r|-m)\n" and printUsage;
}
  


#
# add a new receiver and/or data element to receiver's list
# 
# params
#   data element to add
#
sub addReceiver {

  my $data = shift;

  # ensure a valid reference was passed
  if( ref( $data ) ) {
    # get the ip address for the receiver
    my $ip = $data->receiver();

    # check if a receiver already exists with this ip
    unless( exists $receivers{$ip} ) {
      print "Adding receiver $ip for ", ref( $data ), "\n" if $args->{verbose};
      # create a new receiver
      $receivers{$ip} = new Mwalk::Receiver( $ip, $args );
    }
    # set the data element according to its type
    $receivers{$ip}->add( $data );
  }
}


#
# parse the rtcp data from the dump file
#
sub parseRTCP {

  my $input = new FileHandle( "< $args->{rtcpFile}" ) or die "Could not open RTCP file: $!\n";
  my ($rtcp, $reps) = (undef, 1);

  print "Starting RTCP processing for $args->{rtcpFile}\n" if $args->{verbose};
  print "Enter number of reps:" and $reps = <STDIN> + 0 if $args->{interactive};

  # process the entire input file
  while( ! $input->eof() and $reps ) {
    # create a newly parsed rtcp object
    $rtcp = new Mwalk::RTCP( $input );    

    if( $rtcp->state() > 0 ) {
      # if parsed correctly, add to receiver's data
      addReceiver $rtcp;
    } elsif( $rtcp->state() < 0 ) {
      # count the number of skipped rtcp packets
      $stats{counts}{skip}{rtcp}++;
    } else {
      # quit when last object was left unparsed
      last;
    }
  } continue {
    print "Enter number of reps:" and $reps = <STDIN> + 0 if $args->{interactive} and ! --$reps;
  }
  
  # bark if we didn't finish the file
  print "RTCP file was not completely parsed\n" if $args->{verbose} and ! $input->eof();
  close $input;
}


#
# parse the mtrace data from the dump file
#
sub parseMTrace {  

  my $input = new FileHandle( "< $args->{mtraceFile}" ) or die "Could not open MTrace file: $!\n";
  my ($mtrace, $mset, $reps) = (undef, new Mwalk::TraceSet( $args ), 1);
  
  print "Starting MTrace processing for $args->{mtraceFile}\n" if $args->{verbose};
  print "Enter number of reps:" and $reps = <STDIN> + 0 if $args->{interactive};

  # process the entire input file
  while( ! $input->eof() and $reps ) {
    # create a newly parsed mtrace object, with possibly strict parsing requirements
    $mtrace = new Mwalk::MTrace( $args->{strict} );
    $mtrace->parseLog( $input );
    
    # check if parsed correctly, and try to add to existing trace set
    if( $mtrace->state() > 0 ) {
      unless( $mset->add( $mtrace ) ) {
	# add the combined trace from the current set
	addReceiver $mset->merge();
	# create a new trace set for this mtrace
	$mset = new Mwalk::TraceSet( $args, $mtrace );
      }
      $stats{skips}{none}++;

    } elsif( $mtrace->state() < 0 ) {
      # count the number of skipped traces
      $stats{counts}{skip}{mtrace}++;
      # account for why the mtrace was skipped
      $stats{skips}{form}++, next if $mtrace->error() == $ERR_FORMAT;
      $stats{skips}{cmd}++, next if $mtrace->error() == $ERR_CMD;
      $stats{skips}{hopless}++, next if $mtrace->error() == $ERR_HOPLESS;
      $stats{skips}{start}++, next if $mtrace->error() == $ERR_START;
      $stats{skips}{strict}++, next if $mtrace->error() == $ERR_STRICT;
      $stats{skips}{loop}++, next if $mtrace->error() == $ERR_LOOP;

    } else {
      # quit when last object was left unparsed
      last;
    } 
  } continue {
    print "Enter number of reps:" and $reps = <STDIN> + 0 if $args->{interactive} and ! --$reps;
  }

  # we may have a final set that needs to be merged and stored
  addReceiver $mset->merge();

  # bark if we didn't finish the file
  print "MTrace file was not completely parsed\n" if $args->{verbose} and ! $input->eof();
  close $input;
}



#
# save statistics about data collection
#
sub saveStats {
 
  my ($first, $last, $recvs) = (0, 0, 0);
  my ($byes, $byed, $patches, $patched, $completes, $completed) = (0, 0, 0, 0, 0, 0);

  print "Compiling statistics\n" if $args->{verbose};

  # compute stats for each receiver
  my ($ip, $recv);
  while( ($ip, $recv) = each %receivers ) {
    # compute stats for rtcp and mtrace data
    foreach my $data ("interval", "rtcp", "mtrace") {
      my $len = $recv->size( $data );
      $stats{counts}{min}{$data} = $len if $len < $stats{counts}{min}{$data};
      $stats{counts}{max}{$data} = $len if $len >  $stats{counts}{max}{$data};
      $stats{counts}{with}{$data}++ if $len;
      $stats{counts}{tot}{$data} += $len;
    }
    # track start and stop time of entire session
    $first = $recv->firstTime() if ! $first or $recv->firstTime() < $first;
    $last = $recv->lastTime() if $recv->lastTime() > $last;
    # keep count of byes processed and traces complete and/or patched
    $byes += $recv->byes();
    $byed++ if $recv->byes();
    $patches += $recv->patches();
    $patched++ if $recv->patches();
    $completes += $recv->completes();
    $completed++ if $recv->completes();
    # count the number of receivers
    $recvs++;
  }

  # open a new file for writing if stat file specified
  my $fh = new FileHandle( (($args->{append}) ? ">>" : ">") . " $args->{statFile}" ) 
           or die "Can't open stat file $args->{statFile}: $!\n";
  
  # print runtime arguments
  $args->save( $fh );
  # print runtime times
  print $fh "start:", scalar localtime $args->{start}, "\n";
  print $fh "stop:", scalar localtime $args->{stop}, "\n";
  print $fh "elapsed:", $args->{stop} - $args->{start}, " sec.\n\n";

  # print stats
  print $fh "STATS:\n";
  print $fh "first:$first\nlast:$last\n";
  print $fh "receivers:$recvs\n";
  print $fh "byes:$byes from $byed\npatches:$patches from $patched\n";
  print $fh "complete:$completes from $completed\n";
  # print individual stats for rtcp and mtrace data
  foreach my $data ("interval", "rtcp", "mtrace") {
    print $fh "\n\U$data:\n";
    print $fh "min:$stats{counts}{min}{$data}\nmax:$stats{counts}{max}{$data}\nwith:$stats{counts}{with}{$data}\n";
    print $fh "avg:", (($recvs) ? $stats{counts}{tot}{$data} / $recvs : 0), "\ntot:$stats{counts}{tot}{$data}\n";
    print $fh "skip:$stats{counts}{skip}{$data}\n"
  }
  print $fh "SKIPS:\n";
  foreach my $skip (keys %{$stats{skips}}) {
    print $fh "$skip:$stats{skips}{$skip}\n";
  }

  # close the stats file
  close $fh;
}


#
# save collected data into output file(s)
#
sub saveData {

  my ($with, $without, $mode);
  
  # open tabulation files to record which receivers have mtraces
  $mode = ($args->{append}) ? ">>" : ">";
  $with = new FileHandle( "$mode $args->{dir}receivers.with" ) or die "Couldn't open with receivers file: $!\n";
  $without = new FileHandle( "$mode $args->{dir}receivers.without" ) or die "Couldn't open without receivers file: $!\n";

  # save data for each receiver
  my ($ip, $recv, $fh);
  while( ($ip, $recv) = each %receivers ) {     
    print "Saving data for receiver $ip\n" if $args->{verbose};
    # save receiver data in individual files named by ip  
    $recv->save();
    # save whether this receiver has mtrace data available
    $fh = ($recv->mtraces()) ? $with : $without;
    print $fh $recv->ip(), "\n";
  }

  close $with;
  close $without;
}



#
# enter an interactive mode to allow on-line querying
#
sub goInteractive {

  my ($line, $cmd) = (undef, "");

  print "Entering interactive mode\n" if $args->{verbose};
  # print prompt
  print "\nparse> ";

  # loop around until user exits
  while( $line = <STDIN> ) {
    # append the line to the command
    $cmd .= $line;
    
    # check for line continuation
    print "> " and next if $line =~ m/\\\w*$/;
    # check for exit command
    last if $cmd =~ m/^\w*exit\w*(\\\w*)*$/;
    
    # evaluate command if something passed
    print "error: $@\n" if $cmd !~ m/^\w*$/ and ! defined( eval $cmd );

    # clear command and reprint prompt
    $cmd = "";
    print "parse> ";
  }
}
