#
#     File : MTrace.pm
#   Author : Robert Chalmers
#
# Original : November 22, 1999
#  Revised :
#
#  Content : class module encapsulating an individual mutlicast trace
#

package Mwalk::MTrace;


# import external modules
use Exporter;
use FileHandle;

# set inheritance
@ISA = qw( Exporter );

# export variables to other modules
@EXPORT = qw($ERR_NONE $ERR_FORMAT $ERR_CMD $ERR_HOPLESS $ERR_START $ERR_STRICT $ERR_LOOP);
use vars qw($ERR_NONE $ERR_FORMAT $ERR_CMD $ERR_HOPLESS $ERR_START $ERR_STRICT $ERR_LOOP $ERR_PIPE);
# exported error constants
($ERR_NONE, $ERR_FORMAT, $ERR_CMD, $ERR_HOPLESS, $ERR_START, $ERR_STRICT, $ERR_LOOP, $ERR_PIPE) =
(        0,          -1,       -2,           -3,         -4,          -5,        -6,        -7);


# enforce strictness
use strict qw( vars refs subs );

# parsing state constants
my ($STATE_BAIL, $STATE_START, $STATE_CMD, $STATE_HOPS, $STATE_STATS, $STATE_STOP, $STATE_DONE) =
   (         -1,            0,          1,           2,            3,           4,           5);
# constant pattern for ip address
my $IP = '(?:\d{1,3}\.){3}\d{1,3}';

# member variable structure
my %members = ( state    => $STATE_START,
		error    => $ERR_NONE,
		strict   => 0,
		line     => undef,
	        receiver => "",
		source   => "",
		group    => "",
		gateway  => "",
		type     => "",
	        complete => 0,
		flip     => 0,
		patch    => 0,
		start    => 0,
		stop     => 0,
		command  => "",
		hops     => undef
	      );
# hops array element structure
my %hop = ( ip       => "", 
	    name     => "", 
	    incoming => "", 
	    overall  => 0, 
	    lost     => 0, 
	    sent     => 0, 
	    rate     => 0
	  );


#
# constructor
#
# params
#   class name or object reference
#   whether to be strict about parsing
#
# return
#   new object reference
#
sub new {

  my ($that, $strict) = @_;
  my $class = ref( $that ) || $that;

  # build new object from template
  my $self = { %members };
  bless $self, $class;

  # initialize anonymous arrays
  $self->{hops} = [];
  #initialize strict parsing flag
  $self->{strict} = $strict if defined( $strict ) and $strict;

  return $self;
}



#
# parse a complete mtrace entry from a log file
#
# params
#   self reference
#   reference to an open filehandle 
#
# return
#   whether parse completed successfully (1=success, 0=fail, <0=skip)
#   the return value is also saved in {state}
#
sub parseLog {

  my ($self, $fh) = @_;
  
  # ensure a valid file handle was passed
  ref( $fh ) or die "File handle not defined";

  # reset parsing state
  $self->{state} = $STATE_START;
  
  while( $self->{line} = <$fh> and $self->{state} != $STATE_BAIL ) {

    if( $self->{state} == $STATE_START ) {
      # loop through grabage until we get a starting timestamp
      $self->parseTime( "start" );

    } elsif( $self->{state} == $STATE_CMD ) {
      # parse command, skip trace if receiver not determined
      $self->parseCommand();

    } elsif( $self->{state} == $STATE_HOPS ) {
      # parse hops, skip trace if no hops were recorded or hops were invalid
      $self->parseHops();

    } elsif( $self->{state} == $STATE_STATS ) {
      # parse statistics, get back last line not processed in case we hit the ending timestamp
      $self->parseStatistics( $fh );
    }

    if( $self->{state} == $STATE_STOP ) {
      # loop through grabage until we get a ending timestamp
      $self->parseTime( "stop" ) and last;
    }
  }

  # finalize the parsing by cleaning up the hop array and 
  #   determining whether the trace is complete
  return $self->{state} = $self->parseFinalize();
}


#
# parse a generated mtrace
#
# params
#   self reference
#   open file handle to mtrace output
#
# return
#   whether parsing is complete
#
sub parseLive {

  my ($self, $fh) = @_;

    # ensure a valid file handle was passed
  ref( $fh ) or die "File handle not defined";

  # reset parsing state
  $self->{state} = $STATE_CMD;
  
  while( $self->{line} = <$fh> and $self->{state} != $STATE_BAIL ) {

    if( $self->{state} == $STATE_CMD ) {
      # parse out command block
      $self->parseCommand();

    } elsif( $self->{state} == $STATE_HOPS ) {
      # parse hops, skip trace if no hops were recorded or hops were invalid
      $self->parseHops();

    } elsif( $self->{state} == $STATE_STATS ) {
      # parse statistics
      $self->parseStatistics( $fh );
    }
  }

  # mark the parsing as done if we got hops
  $self->{state} = $STATE_DONE if $self->{state} > $STATE_HOPS;

  # finalize the parsing by cleaning up the hop array and 
  #   determining whether the trace is complete
  return $self->{state} = $self->parseFinalize();
}


#
# parse the start/stop time of the trace
#
# params
#   self reference
#   name of variable to parse (start,stop) 
#
# return
#   whether parsing is complete
#
sub parseTime {

  my ($self, $var) = @_;
  my ($match) = (($var eq "start") ? "START" : "FINISH");

  # ensure a valid line is available
  defined( $self->{line} ) or die "Parse line not defined";

  # loop through garbage until we get a starting timestamp
  if( $self->{line} =~ m/^\# TIMESTAMP (\w+) = (\d+)/o ) {
    # check if the type matched
    if( $1 eq $match ) {
      # save the start time and increment parsing state
      $self->{$var} = $2;
      return ++$self->{state};
    } elsif( $self->{state} == $STATE_STOP ) {
      # just keep on looking for a start, but bail if finish didn't match up
      return $self->setError( $ERR_FORMAT );
    }
  }

  # keep on parsing
  return 0;
}


#
# parse an mtrace command
#
# params
#   self reference
#
# return
#   whether parsing is complete
#
sub parseCommand {

  my $self = shift;

  # ensure a valid line is available
  defined( $self->{line} ) or die "Parse line not defined";

  # bail if we see a timestamp
  return $self->setError( $ERR_FORMAT ) if $self->{line} =~ m/TIMESTAMP/o;
  # break at query line
  if( $self->{line} =~ m/^Querying/o ) {
    # error if we didn't parse the command-line
    if( $self->receiver ) {
      return ++$self->{state};
    } else {
      return $self->setError( $ERR_CMD );
    }
  }

  # parse the mtrace command line for its parameters
  #   ignore commands with a single ip as a parameter
  #   these were used simply to determine the gateway
  if( $self->{line} =~ m/^mtrace (?:-n?([sg]))? *($IP) ($IP)(?: ($IP))?(?: ($IP))?/o ) {
    # save the command line w/o trailing newline
    $self->{command} = $self->{line};
    chomp $self->{command};

    # this means we didn't get a switch
    if( ! defined( $1 ) ) {
      $self->{type} = "normal";
      $self->{source} = $2;
      $self->{receiver} = $3;
      $self->{group} = $4;
      # otherwise we should have a -s or -g switch
    } elsif( $1 eq "s" ) {
      $self->{type} = "source";
      $self->{receiver} = $2;
      $self->{source} = $3;
      $self->{group} = $4 if defined( $4 );	
      $self->{flip} = 1;
    } elsif( $1 eq "g" ) {
      $self->{type} = "gateway";
      $self->{gateway} = $2;
      $self->{source} = $3;
      if( defined( $5 ) ) {
	$self->{group} = $4;
	$self->{receiver} = $5;
      } else {
	$self->{receiver} = $4;
      }
    }
  }

  # need to continue parsing command
  return 0;
}


#
# parse the hops of an mtrace
#
# params
#   self reference
#
# return
#   whether parsing is complete
#
sub parseHops {

  my $self = shift;
  my %newhop = (%hop);
  my ($hop, $name, $ip);

  # ensure a valid line is available
  defined( $self->{line} ) or die "Parse line not defined";

  # check for hop line
  if( ($hop, $name, $ip) = ($self->{line} =~ m/^ *-?(\d+)[ *?]*([-\w.]+)? *(?:\(($IP)\))?/o ) ) {
    # fetch ip from one of two possible position depending on presence of -n switch
    if( defined( $name ) ) {
      if( ! defined( $ip ) and $name =~ m/^$IP$/o ) {
	$newhop{ip} = $name;
      } else {
	$newhop{ip} = $ip if defined( $ip );
	$newhop{name} = $name;
      }
    } else {
      $newhop{ip} = $ip if defined( $ip );
    }
    
    # place hash on hops array according to index
    $self->{hops}[$hop] = \%newhop;

  } elsif( $self->{line} !~ m/^Resuming|^\s*$|^Route with/o ) {
    # skip resuming statements otherwise quit
    # ensure we have some hops
    return $self->setError( $ERR_HOPLESS ) unless @{$self->{hops}};
    # ensure source traces start with source and others start with receiver
    return $self->setError( $ERR_START ) unless $self->isValid();

    # all seems well
    return ++$self->{state};
  }

  # need to continue parsing hops
  return 0;
}


#
# parse the statistics of an mtrace
#
# params
#   self reference
#   reference to open file handle
#
# return
#   whether parsing is complete
#
sub parseStatistics {

  my ($self, $fh) = @_;
  my (%hops, $in, $out);

  # ensure a valid file handle was passed
  ref( $fh ) or die "File handle not defined";

  # create a hash for hops keyed by ip, skipping duplicates
  foreach my $hop (@{$self->{hops}}) {
    return $self->setError( $ERR_FORMAT ) unless defined( $hop );
    $hops{$hop->{ip}} = $hop if ! exists $hops{$hop->{ip}};
  }
 
  while( defined( $self->{line} ) ) {
    # stop if we hit the end of stats or the ending timestamp
    return ++$self->{state} if $self->{line} =~ m/^  Receiver|TIMESTAMP/o;

    # assume we're working with stats if line starts with an ip or down-arrow
    if( $self->{line} =~ m/^($IP)|(?: +v)/o ) {
      # check if we got an ip
      if( defined( $1 ) ) {
	# set $out then $in, skip to ahead to next line
	defined( $out ) and $in = $out;
	$out = $1;
	next;
	
      # check if we have an interface to work with
      } elsif( defined( $out ) ) {
	# save the incoming interface
	$hops{$out}{incoming} = $in if defined( $in ); 
	# check for statistics
	if( $self->{line} =~ m/ttl *\d+ +(\d+) pps +([-\d?]+)\/(\d+) +(?:= *\d+%)? +(\d+) pps/o ) {
	  $hops{$out}{overall} = $1;
	  $hops{$out}{lost} = $2;
	  $hops{$out}{sent} = $3;
	  $hops{$out}{rate} = $4;
	}
      }
    }    
    # clear interface values
    $in = $out = undef;

  } continue {
    # read the next line
    $self->{line} = <$fh>;
  }

  # nothing left to parse
  return 1;
}


#
# cleanup the parsed data and determine if trace is complete
#
# params
#   self reference
#
# return
#   whether parse completed successfully (0=fail, 1=success, -1=skip)
#
sub parseFinalize {

  my $self = shift;

  # sanity check - skip if we bailed on this trace, or fail if we didn't complete it
  return $self->{state} if $self->{state} < $STATE_START;
  return 0 if $self->{state} < $STATE_DONE;

  my (@hops, %hops);
  my ($h, $ok, $hop) = (0, 0, "");
  # remove empty and repeating hops and look for loops
  #   throw out traces that have loops or any problem if running strict
  while( $h < @{$self->{hops}} ) {
    my $ip = $self->{hops}[$h]{ip};

    # skip empty and repeating hops
    next if ! $ip or $ip eq $hop;
    # bail if we see a hop again (a loop exists in the trace)
    return $self->setError( $ERR_LOOP ) if exists $hops{$ip};

    # this entry seems ok, add it to our new list and mark it for the future
    push @hops, $self->{hops}[$h];
    $hops{$ip} = $hop = $ip;
    # mark this pass as ok
    $ok++;
  } continue {
    # trash this mtrace if we had a problem and we're being real strict
    return $self->setError( $ERR_STRICT ) unless $ok or ! $self->{strict};
    # increment our count and clear the ok flag for the next round
    $h++; $ok = 0;
  }

  # assign hops back or reverse the hops if a source query was performed
  if( $self->{flip} ) {
    $self->{hops} = [reverse @hops];
    $self->{flip} = 0;
  } else {
    $self->{hops} = \@hops;
  } 
  
  # determine whether the trace is complete
  #   does the first hop match the receiver and the last the source ?
  $self->isComplete();

  # check if we have more than a single hop (receiver or source)
  #   otherwise, things seem to have gone okay
  return (@{$self->{hops}} > 1) ? 1 : $self->setError( $ERR_HOPLESS);
}


#
# determine and set whether the trace is complete
#
# params
#   self reference
#
# return 
#   whether the trace is complete
#
sub isComplete {
  
  my $self = shift;

  # determine whether the trace is complete
  #   does the first hop match the receiver and the last the source ?
  return $self->{complete} = ($self->{hops}[0]{ip} eq $self->{receiver} and 
                              $self->{hops}[$#{$self->{hops}}]{ip} eq $self->{source});
}


#
# determine and whether the trace is valid
# an mtrace is valid only if it has a receiver/source that is also the first hop
#
# params
#   self reference
#
# return 
#   whether the trace is valid
#
sub isValid {
  
  my $self = shift;

  # ensure a source and receiver is defined
  return 0 unless $self->{source} and $self->{receiver};

  # determine whether the trace is valid
  if( $self->{flip} ) {
    # does the first hop match the source
    return $self->{hops}[0]{ip} eq $self->{source}
  } else {
    # does the first hop match the receiver
    return $self->{hops}[0]{ip} eq $self->{receiver};
  }
}


#
# check the trace for empty hops and loops
# this is similar to parseFinalize() except that it doesn't try to 
# repair the trace at all
#
# params
#   self reference
#
# return
#   whether this trace has problems
#
sub isTrouble {

  my $self = shift;
  my %hops;

  return $self->setError( $ERR_FORMAT ) unless $self->isValid();

  # look for empty hops and loops
  for( my $h = 0; $h < @{$self->{hops}}; $h++ ) {
    my $ip = $self->{hops}[$h]{ip};

    # check for problems
    return $self->setError( $ERR_STRICT ) if ! $ip;
    return $self->setError( $ERR_LOOP ) if exists $hops{$ip};

    # this entry seems ok, track it
    $hops{$ip} = $ip;
  }

  return 0;
}


#
# set the state to bail on this mtrace and set the error code appropriately
#
# params
#   self reference
#   error value to set
#
# return
#   errored state
#
sub setError {

  my ($self, $error) = @_;

  $self->{error} = $error;
  return $self->{state} = $STATE_BAIL;
}


#
# write the processed mtrace out to file
#
# params
#   self reference
#   name of file to save to or open file handle
#
sub save {

  my ($self, $file) = @_;
  my $fh;

  # open the save file unless handle was provided
  if( ref( $file ) ) {
    $fh = $file; 
  } else {
    $fh = new FileHandle( ">> $file" ) or die "Can't open save file $file: $!\n";
  }

  # output each part of the mtrace data
  # print the trace indicator with completion flag source/reciver/group and start/stop times
  print $fh "TRACE:", ($self->{complete}) ? "complete" : "partial", "\n";
  print $fh "receiver:$self->{receiver}\nsource:$self->{source}\n";
  print $fh "group:$self->{group}\ngw:$self->{gateway}\n";
  print $fh "type:$self->{type}\npatch:", ($self->{patch}) ? "yes" : "no", "\n";
  print $fh "start:$self->{start}\nstop:$self->{stop}\n";
  print $fh "command:$self->{command}\n";
  print $fh "hops:", scalar @{$self->{hops}}, "\n";
  # print the mtrace hop data
  for( my $i = 0; $i < @{$self->{hops}}; $i++ ) {
    my $line = $self->{hops}[$i];
    # each line of hops is actually a hash
    printf $fh "%3d [%s] [%s] [%s] [%d] [%s/%d] [%d]\n", $i, 
    $line->{ip}, $line->{name}, $line->{incoming}, 
    $line->{overall}, $line->{lost}, $line->{sent}, $line->{rate}; 
  }
  print $fh "\n";

  # close the file if we opened it
  close $fh unless ref( $file );
}



#
# class destructor
#
# params
#   self reference
#
sub DESTROY {

  my $self = shift;

  #print "Destroying MTrace for $self->{receiver}.\n";
}



#
# provide access to member variables as automatic methods
#
# params
#   self reference
#
sub AUTOLOAD {

  my $self = shift;
  my $type = ref( $self ) || die "Access method called without object reference: $!";
  my $name = $Mwalk::MTrace::AUTOLOAD;

  # strip fully-qualified protion of method
  $name =~ s/.*://;
  # ensure member exists
  exists $self->{$name} or die "Access method called on non-existent member: $name";

  # set or get member variable
  if( @_ ) {
    return $self->{$name} = shift;
  } else {
    return $self->{$name};
  }
}
