/* 
     File : Builder.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A runnable class that builds the tree from mtraces and receiver
            interval information.
             
  $Id: Builder.java,v 1.7 2000/11/18 04:49:27 robertc Exp $
*/

package mwalk.app;


import java.util.Date;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.io.File;
import java.io.FileReader;
import java.io.BufferedReader;

import mwalk.core.*;
import mwalk.visitor.*;
import mwalk.util.BuildException;
import mwalk.util.ConfigException;
import mwalk.util.VisitException;


/**
 * A runnable class that builds the tree from mtraces and receiver
 * interval information.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class Builder {

    /** Number of other sources seen when building paths */
    protected int otherSources = 0;
    /** Number of primary links in the tree */
    protected int treeLinks = 0;
    /** Number of total edges in graph */
    protected int graphEdges = 0;


    /**
     * Default constructor.
     */
    public Builder() {}


    /**
     * Build tree from supplied logs.
     *
     * @exception <code>BuildException</code> if an error occurred during build
     */
    public Tree build() throws BuildException {

	Tree tree;
	Date start = new Date();

	if( Config.readTree != null ) {
	    // build a tree from a saved version
	    tree = Tree.load( Config.readTree );
	    Config.source = tree.getSource().getIP();
	}
	else {
	    // build a new tree for a particular source
	    tree = new Tree( Config.source );
	    // default directory list to local directory
	    if( Config.dirs.length == 0 )
		Config.dirs = new String[] { "./" };
	}

	// read the stats files to determine the range of time for this simulation
	readStats( tree );

	// add unique receivers to tree
	addReceivers( tree );

	// add path information to tree
	addPaths( tree );

	// save tree out to file
	if( tree.isUpdated() && Config.writeTree != null )
	    tree.save( Config.writeTree );

	// clear pathless receivers
	cleanReceivers( tree );
	// merge the period trees for each node
	mergePeriods( tree );
	// set primary paths
	markPrimaryPaths( tree );

	Config.verbose( "\nTree built in " + (((new Date()).getTime() - start.getTime()) / 1000) + " sec." );
	Config.verbose( "Contains " + tree.receivers() + " receivers, " + tree.nodes() + " nodes, " + treeLinks + " links, " + graphEdges + " edges" );
	Config.verbose( "Found " + otherSources + " traces with other sources" );

	return( tree );
    }


    /**
     * Read statistics file to extract the session time for the given session.
     *
     * @param <code>Tree</code> active tree
     * @exception <code>BuildException</code> if session time not available
     */
    public void readStats( Tree tree ) throws BuildException {

	File file;
	String line;

	for( int i = 0; i < Config.dirs.length; i++ ) {
	    // check if stats file exists
	    file = new File( Config.dirs[i] + "stats" );
	    if( file.exists() )
		try {
		    Config.verbose( "Reading stats: " + file.getPath() );		    
		    BufferedReader stats = new BufferedReader( new FileReader( file ) );
		    long start = 0, stop = 0;
		    
		    // parse stats file for first and last seen activity
		    while( (line = stats.readLine()) != null )
			if( line.startsWith( "first" ) )
			    start = Long.parseLong( line.substring( line.indexOf( ':' ) + 1 ) );
			else if( line.startsWith( "last" ) ) {
			    stop = Long.parseLong( line.substring( line.indexOf( ':' ) + 1 ) );
			    break;
			}
		    
		    // store the start and stop range for any simulations
		    if( start > 0 && stop > 0 )
			tree.setRange( start, stop );
		    else
			throw new BuildException( "no start or stop in stats file: " + file );
		    
		} catch( Exception e ) {
		    throw new BuildException( "error parsing stats file: " + file, e );
		}
	}
    } 


    /** 
     * Add receivers to tree from parsed RTCP logs.
     *
     * @param <code>Tree</code> active tree
     * @exception <code>BuildException</code> if receivers could not be added
     */
    public void addReceivers( Tree tree ) throws BuildException {

	File file;
	String ip;

	for( int i = 0; i < Config.dirs.length; i++ ) {
	    // check if a receiver list is present
	    file = new File( Config.dirs[i] + "receivers.with" );
	    if( file.exists() )
		try {
		    BufferedReader with = new BufferedReader( new FileReader( file ) );
		    
		    // parse the receivers file listing only those that have corresponding mtraces
		    while( (ip = with.readLine()) != null && ip.trim().length() > 0 ) 
			if( ! tree.isNode( ip ) ) {
			    Config.verbose( "Adding receiver " + ip );
			    tree.addReceiver( new Receiver( ip ) );
			}
		    
		} catch( Exception e ) {
		    throw new BuildException( "error adding receiver: " + file, e );
		}
	}
    }


    /** 
     * Add paths to tree from parsed mtrace logs.
     *
     * @param <code>Tree</code> active tree
     * @exception <code>BuildException</code> if paths could not be added
     */
    public void addPaths( Tree tree ) throws BuildException {

	File file;

	for( Enumeration recvs = tree.getReceivers(); recvs.hasMoreElements(); )
	    try {
		// get list of all receivers
		Receiver recv = (Receiver)recvs.nextElement();
		
		// loop through each data set
		for( int i = 0; i < Config.dirs.length; i++ ) {
		    // check if interval file for this receiver exists
		    file = new File( Config.dirs[i] + recv.getIP() + ".interval" );
		    if( file.exists() )
			// load interval ranges for receiver
			try {
			    Config.verbose( "Processing intervals for " + recv.getIP() + ": " + file.getPath() );
			    BufferedReader intervals = new BufferedReader( new FileReader( file ) );

			    recv.loadIntervals( intervals );
			    tree.markUpdated();
			    
			} catch( BuildException be ) {
			    throw be;
			} catch( Exception e ) {
			    throw new BuildException( "error adding interval in " + file, e );
			}

		    // check if mtrace file for this receiver exists
		    file = new File( Config.dirs[i] + recv.getIP() + ".mtrace" );
		    if( file.exists() )
			// parse each mtrace and build the path in the tree from receiver to source
			try {
			    Config.verbose( "Processing mtraces for " + recv.getIP() + ": " + file );
			    BufferedReader traces = new BufferedReader( new FileReader( file ) );
			    
			    while( traces.ready() ) 
				try {
				    MTrace trace = new MTrace( traces );
				    // use only complete traces
				    if( trace.parsed && trace.complete ) {
					tree.walkUp( new TraceVisitor( trace ), recv );
					tree.markUpdated();
				    }
				    
				} catch( ConfigException ce ) {
				    Config.verbose( "Found trace for different source" );
				    otherSources++;
				}
			} catch( BuildException be ) {
			    throw be;
			} catch( Exception e ) {
			    throw new BuildException( "error adding path in " + file, e );
			}		    
		}
	    } catch( NoSuchElementException nse ) {}
    }


    /** 
     * Remove receivers from the tree if they have no path back to the source.
     *
     * @param <code>Tree</code> active tree
     */
    public void cleanReceivers( Tree tree ) {

	Config.verbose( "\nCleaning pathless receivers" );
	// get a list of all receivers
	TreeNode[] recvs = TreeNode.toNodeArray( tree.getReceivers(), tree.receivers() );
	// remove any receivers without a parent (no path to source)
	for( int r = 0; r < recvs.length; r++ )
	    if( recvs[r].parents() == 0 )
		tree.delNode( recvs[r] );
    }


    /** 
     * Merge overlapping active periods to reduce the size of the perio tables.
     *
     * @param <code>Tree</code> active tree
     * @exception <code>BuildException</code> if merge failed
     */
    public void mergePeriods( Tree tree ) throws BuildException {

	try {
	    // merge the period tables and bound times based on child tables
	    Config.verbose( "\nMerging period tables" );
	    tree.walkDown( new MergeVisitor() );

	    tree.clearData();

	    // set the earliest start time for each receiver along the earliest path
	    Config.verbose( "\nSetting earliest receiver times" );
	    tree.walkUp( new JoinVisitor() );

	    tree.markMerged();
	    
	} catch( VisitException ve ) {
	    throw new BuildException( "error merging period tables", ve );
	}
    }


    /** 
     * Mark primary paths for each receiver.
     * The primary path is the path that is active for the longest time 
     * throughout the duration of the session.
     *
     * @param <code>Tree</code> active tree
     * @exception <code>BuildException</code> if merge failed
     */
    public void markPrimaryPaths( Tree tree ) throws BuildException {

	try {
	    PrimaryPathVisitor visitor = new PrimaryPathVisitor();

	    // set the primary paths for each receiver
	    Config.verbose( "\nMarking primary paths" );
	    tree.walkUp( visitor );

	    // save tree link and graph egde counts
	    treeLinks = visitor.countLinks();
	    graphEdges = visitor.countEdges();
	    
	} catch( VisitException ve ) {
	    throw new BuildException( "error setting primary paths", ve );
	}
    }


    /**
     * Main application method.
     *
     * @param <code>String[]</code> command line arguments.
     */
    public static void main( String args[] ) {

	try {
	    // parse the command-line arguments
	    Config.parseArgs( args );
	    // setup a defult save file if not specified
	    if( Config.writeTree == null )
		Config.writeTree = "tree";

	    // build the tree and save it out
	    Builder builder = new Builder();
	    builder.build();

	} catch( Exception e ) {
	    e.printStackTrace();
	    System.exit( -1 );
	}
    }
}
	
