/* 
     File : Receiver.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A specialized TreeNode class that represents a receiver in 
            the multicast tree.
             
  $Id: Receiver.java,v 1.4 2000/11/18 04:49:28 robertc Exp $
*/

package mwalk.core;


import java.util.Hashtable;
import java.io.BufferedReader;

import mwalk.util.BuildException;


/**
 * A specialized TreeNode class that represents a receiver in
 * the multicast tree.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class Receiver extends TreeNode {

    /** Serialized version ID */
    static final long serialVersionUID = -9064699466070338310L;

    /** Period table representing receivers activity */
    protected PeriodTable intervals = new PeriodTable();


    /**
     * Constructor.
     *
     * @param <code>String</code> the IP address of the reciever
     */
    public Receiver( String ip ) {

	super( ip );

	parents = new Hashtable();
    }


    /**
     * Return the receiver's activity table.
     *
     * @return <code>PeriodTable</code> receiver's activity table
     */
    public PeriodTable getIntervals() {

	return( intervals );
    }


    /**
     * Read the activity table from parsed RTCP logs.
     *
     * @param <code>BufferedReader</code> reader associated to log
     * @exception <code>BuildException</code> if parsing failed
     */
    public void loadIntervals( BufferedReader br ) throws BuildException {

	intervals.load( br );
    }


    /**
     * Check whether receiver is explicitly marked active.
     *
     * @return <code>boolean</code> whether receiver is active
     */
    public boolean isActive() {

	return( active );
    }


    /**
     * Check whether receiver is active for a given timestamp.
     *
     * @param <code>long</code> timestamp to check
     * @return <code>boolean</code> whether receiver is active
     */
    public boolean isActive( long time ) {

	return( intervals.check( time ) );
    }


    /**
     * Check whether the receiver's earliest active period has been reached.
     *
     * @param <code>long</code> timestamp to check
     * @return <code>boolean</code> whether receiver has been active
     */
    public boolean hasArrived( long time ) {

	return( intervals.getEarliest() <= time );
    }
}
