/* 
     File : Tree.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A class representing the set of possible multicast trees.
             
  $Id: Tree.java,v 1.4 2000/11/18 04:49:28 robertc Exp $
*/

package mwalk.core;


import java.util.Vector;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.NoSuchElementException;
import java.io.Serializable;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import mwalk.visitor.UpVisitor;
import mwalk.visitor.DownVisitor;
import mwalk.util.BuildException;
import mwalk.util.VisitException;


/**
 * A class representing the set of possible multicast trees.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class Tree implements Serializable {

    static final long serialVersionUID = 3926210686248469450L;

    protected Source source = null;
    protected Hashtable nodes = new Hashtable();
    protected Hashtable receivers = new Hashtable();
    protected long start = Long.MAX_VALUE, stop = Long.MIN_VALUE;
    protected boolean merged = false;

    protected transient boolean updated = false;


    public Tree( String ip ) {

	source = new Source( ip );
	nodes.put( source.getIP(), source );
    }

    public static Tree load( String file ) throws BuildException {

	Config.verbose( "Reading tree from file: " + file );

	try {
	    ObjectInputStream is = new ObjectInputStream( new FileInputStream( file ) );
	    Tree tree = (Tree)is.readObject();

	    // clear the update flag
	    tree.updated = false;
	    return( tree );
	    
	} catch( Exception e ) {
	    throw new BuildException( "error reading tree from file: " + file, e );
	}
    }

    public void save( String file ) throws BuildException {

	Config.verbose( "Writing tree to file: " + file );
	
	try {
	    ObjectOutputStream os = new ObjectOutputStream( new FileOutputStream( file ) );
	    os.writeObject( this );
	    
	} catch( Exception e ) {
	    throw new BuildException( "error writing tree to file: " + file, e );
	}
    }


    public boolean isMerged() {

	return( merged );
    }


    public void markMerged() {

	 merged = true;
	 updated = true;
    }

    public boolean isUpdated() {

	return( updated );
    }

    public void markUpdated() {

	updated = true;
    }

    public long getStart() {

	return( start );
    }

    public long getStop() {
	
	return( stop );
    }

    public void setRange( long start, long stop ) {
	
	if( start < this.start || stop > this.stop ) {
	    // set the minimum start time and maximum stop time
	    this.start = Math.min( this.start, start );
	    this.stop = Math.max( this.stop, stop );

	    markUpdated();
	}
    }


    public boolean isSource( String ip ) {

	return( source.equals( ip ) );
    }

    public Source getSource() {

	return( source );
    }

    public boolean isReceiver( String ip ) {

	return( receivers.get( ip ) != null );
    }

    public Receiver getReceiver( String ip ) {

	return( (Receiver)receivers.get( ip ) );
    }

    public Enumeration getReceivers() {

	return( receivers.elements() );
    }

    public int receivers() {

	return( receivers.size() );
    }

    public boolean isNode( String ip ) {

	return( nodes.get( ip ) != null );
    }

    public TreeNode getNode( String ip ) {

	return( (TreeNode)nodes.get( ip ) );
    }

    public Enumeration getNodes() {

	return( nodes.elements() );
    }

    public int nodes() {

	return( nodes.size() );
    }


    public TreeNode addNode( TreeNode node ) {

	if( nodes.containsKey( node.getIP() ) )
	    return( (TreeNode)nodes.get( node.getIP() ) );
	else {
	    markUpdated();
	    nodes.put( node.getIP(), node );
	    return( node );
	}
    }

    public Receiver addReceiver( Receiver recv ) {

	if( receivers.containsKey( recv.getIP() ) )
	    return( (Receiver)receivers.get( recv.getIP() ) );
	else {
	    markUpdated();
	    recv = (Receiver)addNode( recv );
	    receivers.put( recv.getIP(), recv );
	    return( recv );
	}
    }


    public void delNode( TreeNode node ) {

	// can't remove the source
	if( node instanceof Source )
	    return;

	// remove node as child from each parent
	TreeNode[] parents = node.getParents();
	for( int p = 0; parents != null && p < parents.length; p++ ) 
	    parents[p].delChild( node );

	// remove node as parent from eachb child
	TreeNode[] children = node.getChildren();
	for( int c = 0; children != null && c < children.length; c++ )
	    children[c].delParent( node );

	// remove node from node list and receiver list if appropriate
	nodes.remove( node.getIP() );
	receivers.remove( node.getIP() );
    }


    public void clearData() {

	for( Enumeration enodes = nodes.elements(); enodes.hasMoreElements(); )
	    try {
		((TreeNode)enodes.nextElement()).clearData();

	    } catch( Exception e ) {}
    }


    public void walkDown( DownVisitor visitor ) throws VisitException {

	Config.verbose( visitor, "Starting walk down at " + source.getIP() );
	walkDown( visitor, source );
    }


    public Object walkDown( DownVisitor visitor, TreeNode node ) throws VisitException {
       
	// base case is the receiver
	if( node instanceof Receiver ) {
	    Config.verbose( visitor, "Hit receiver " + node.getIP() );
	    return( visitor.visitDown( this, node ) );
	}

	// allow the visitor to run before it children our visited
	Object result = null;
	if( visitor.prefix() )
	    result = visitor.visitDown( this, node );

	// get a list of children to visit (may be a subset of all possible)
	TreeNode kids[] = visitor.getChildren( this, node );
	Vector cresults = new Vector();

      	for( int k = 0; k < kids.length; k++ ) {
	    Config.verbose( visitor, "Descending to child " + kids[k].getIP() );
	    Config.pause( visitor );

	    // recurse to each child collecting their responses
	    Object cresult = walkDown( visitor, kids[k] );
	    if( cresult != null && visitor.postfix() )
		cresults.add( cresult );
	}
	
	// allow the visitor to run after all children have been visited
	//  the children's responses are passed in
	if( visitor.postfix() )
	    result = visitor.visitDown( this, node, cresults );

	// return a result to your parent
	return( result );
    }



    public void walkUp( UpVisitor visitor ) throws VisitException {

	// walk up from receiver to source for each receiver
	for( Enumeration recvs = visitor.getReceivers( this ); recvs.hasMoreElements(); )
	    try {
		Receiver recv = (Receiver)recvs.nextElement();
		if( visitor.nextReceiver( this, recv ) ) {
		    Config.verbose( visitor, "Starting walk up at " + recv.getIP() );
		    walkUp( visitor, recv );
		}
	    } catch( NoSuchElementException nse ) {
		Config.verbose( "Ran out of receivers in walkUp()" );
	    }
    }	    


    public Object walkUp( UpVisitor visitor, TreeNode node ) throws VisitException {

	// base case is the source
	if( node instanceof Source ) {
	    Config.verbose( visitor, "Hit source " + node.getIP() );
	    return( visitor.visitUp( this, node ) );
	}
	
	// allow the visitor to run before its children are visited
	Object result = null;
	if( visitor.prefix() )
	    result = visitor.visitUp( this, node );

	// collect a list of parents to visit (may be a subset of all possible)
	TreeNode parents[] = visitor.getParents( this, node );
	Vector cresults = new Vector();

      	for( int p = 0; p < parents.length; p++ ) {
	    Config.verbose( visitor, "Ascending to parent " + parents[p].getIP() );
	    Config.pause( visitor );
	    
	    // recurse to each parent, collecting their responses
	    Object cresult = walkUp( visitor, parents[p] );
	    if( cresult != null && visitor.postfix() )
		cresults.add( cresult );
	}
	
	// allow visitor to run after all parents have been visited
	if( visitor.postfix() )
	    result = visitor.visitUp( this, node, cresults );

	// return a result to your child
	return( result );
    }
}
