/* 
     File : BasicMetricVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A basic implementation of a visitor that gathers metric
            data and saves it out to a file. 
             
*/

package mwalk.visitor;


import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Config;
import mwalk.util.MetricCount;
import mwalk.util.VisitException;


/**
 * A basic implementation of a visitor that gathers metric
 * data and saves it out to a file.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class BasicMetricVisitor extends MetricVisitor {

    private static final String DEF_FILENAME = "tree-metric.basic";

    protected boolean optRun = false;
    protected boolean optMulti = false;


    public BasicMetricVisitor() {

	super( DEF_FILENAME );
    }

    public BasicMetricVisitor( boolean verbose ) {

	super( verbose );

	setFilename( DEF_FILENAME );
    }

    public BasicMetricVisitor( String filename ) {

	super( filename );
    }

    public BasicMetricVisitor( long periods ) {

	super( DEF_FILENAME, periods, false );
    }

    public BasicMetricVisitor( String filename, long periods, boolean verbose ) {

	super( filename, periods, verbose );
    }


    public String getFilename() {

	return( filename + ((optMulti) ? ".mopt" : "") );
    }

    protected void printHeader( Tree tree ) {

	super.printHeader( tree );

	ps.println( "# simulation start:" + tree.getStart() + " stop:" + tree.getStop() + " periods:" + periods );
	ps.println( "# multicast optimized:" + optMulti );
	ps.println( "#" );
	ps.println( "# Time\tRecvs\tUlinks\tMlinks\tMetric\tM est.\tLu\tLm/Lu\tR^k" );
    }


    public boolean again( Tree tree ) {

	// when again stops for non-optMulti, reset and run for optMulti if desired
	if( ! super.again( tree ) ) {
	    if( ! optMulti && optRun  ) {
		optMulti = true;
		init( tree );
	    } else
		return( false );
	}

	return( true );
    }


    protected MetricCount doVisit( Tree tree, TreeNode node, Vector list ) {

	MetricCount count = new MetricCount();
	int klinks = 0;
	
	// handle counts from each child link
	for( Enumeration kids = list.elements(); kids.hasMoreElements(); ) 
	    try {
		MetricCount kcount = (MetricCount)kids.nextElement();
		if( kcount.receivers > 0 ) {
		    Config.verbose( this, "Got a count from child: " + kcount.toString() );
		    count.add( kcount );
		    klinks++;
		} else
		    Config.verbose( this, "Got an empty count from child." );
		
	    } catch( NoSuchElementException nse ) {}
	
	// add ourselves into the pot
	return( addLinks( count, klinks ) );
    }


    protected MetricCount addLinks( MetricCount count, int mlinks ) {

	// add ourselves into the pot (with optional optimization for multicast)
	if( count.receivers > 0 ) {
	    Config.verbose( this, "Adding counts for " + count.receivers + " receivers." );
	    count.ucastLinks += count.receivers;
	    count.mcastLinks += (optMulti) ? 1 : mlinks;
	}

	return( count );
    }

	
    protected void process( Tree tree ) throws VisitException {

	try {
	    // compute the final metric data from the counts	    
	    MetricCount count = (MetricCount)tree.getSource().data;
	    double metric = 0, mest = 0, aucast = 0, cratio = 0, cpower = 0;
	    if( count.receivers > 0 ) {
		metric = 1.0 - (double)count.mcastLinks / (double)count.ucastLinks;
		mest = 1.0 - Math.pow( count.receivers, -0.2 );
		aucast = (double)count.ucastLinks / (double)count.receivers;
		cratio = (double)count.mcastLinks / aucast;
		cpower = Math.pow( count.receivers, 0.8 );
	    }
		
	    // build a buffer with each field separated by tabs
	    StringBuffer buf = new StringBuffer( 128 );
	    char tab = '\t';
	    buf.append( time ).append( tab ).append( count.toString() ).append( tab );
	    buf.append( metric ).append( tab ).append( mest ).append( tab );
	    buf.append( aucast ).append( tab ).append( cratio ).append( tab ).append( cpower );
	    
	    // convert buffer to one string
	    String line = buf.toString();
	    Config.verbose( this, line );
	    printData( tree, line );

	} catch( Exception e ) {
	    throw new VisitException( "error printing metric data at time " + time, e );
	}
    }
}
