/* 
     File : JoinVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A visitor used to mark the initial join time along a receiver's
            earliest path.
             
*/

package mwalk.visitor;


import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;
import mwalk.core.Link;


/**
 * A visitor used to mark the initial join time along a receiver's
 * earliest path.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class JoinVisitor extends AbstractVisitor implements UpVisitor {

    protected long earliest = Long.MAX_VALUE;
    protected String ip = "";

    public JoinVisitor() {}


    public void reset( Tree tree ) {

	super.reset( tree );

	// reset the earliest to infinity
	earliest = Long.MAX_VALUE;
    }


    public boolean nextReceiver( Tree tree, Receiver recv ) {

	reset( tree );
	// set earliest to be the first time this receiver was heard from
	earliest = recv.getIntervals().getEarliest();
	
	return( true );
    }


    public TreeNode[] getParents( Tree tree, TreeNode node ) {

	// get a list of available parents
	TreeNode parents[] = super.getParents( tree, node );
	TreeNode parent = null;
	Link plink = null;
	long early = Long.MAX_VALUE;
	
	// look for parent with earliest period entry
	for( int p = 0; p < parents.length; p++ ) {
	    // get the link from parent to node
	    Link link = parents[p].getChildLink( node );
	    long first = link.activity.getEarliest();
	    if( first < early ) {
		// save the early time, which parent and which link
		early = first;
		parent = parents[p];
		plink = link;
	    }
	}
	
	// return the parent, setting it's earliest period even earlier
	if( parent != null ) {
	    plink.activity.setEarliest( earliest );
	    if( verbose() )
		plink.activity.print();
	    return( new TreeNode[] { parent } );
	}
	else
	    return( new TreeNode[0] );
    }
}
