/* 
     File : LinkCountVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A visitor used to count the number of links for each receiver 
            along the active tree.
             
*/

package mwalk.visitor;


import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;
import mwalk.core.Config;
import mwalk.util.VisitException;


/**
 * A visitor used to count the number of links for each receiver
 * along the active tree.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class LinkCountVisitor extends PrintVisitor implements DownVisitor {

    private static final String DEF_FILENAME = "tree-depth";


    public LinkCountVisitor() {

	super( DEF_FILENAME );
    }

    public LinkCountVisitor( boolean verbose ) {

	super( DEF_FILENAME, verbose );
    }

    public LinkCountVisitor( String filename ) {

	super( filename );
    }

    public LinkCountVisitor( String filename, boolean verbose ) {

	super( filename, verbose );
    }


    public boolean prefix() {

	return( true );
    }


    protected void printHeader( Tree tree ) {

	super.printHeader( tree );

	ps.println( "#" );
	ps.println( "# receiver\tlinks" );
    }

    public boolean init( Tree tree ) {

	// set source at depth zero
	tree.getSource().data = new Integer( 0 );

	return( super.init( tree ) );
    }

    public Object visitDown( Tree tree, TreeNode node ) throws VisitException {

	// check if node has been seen
	if( ! haveSeen( node ) ) {	    
	    int depth;
	    try {
		// figure out our current depth
		depth = ((Integer)node.data).intValue();
		
	    } catch( Exception e ) {
		throw new VisitException( "node did not have current depth", e );
	    }
	    
	    // print out depth for receiver, else set children's depth
	    if( node instanceof Receiver ) {
		String line = node.getIP() + "\t" + depth;
		
		Config.verbose( this, line );
		printData( tree, line );

	    } else {
		// get a list of active children and set their depth one greater than ours
		TreeNode[] children = node.getActiveChildren( eval );
		depth++;
		
		for( int c = 0; c < children.length; c++ )
		    children[c].data = new Integer( depth );
	    }
	}

	return( null );
    }
}
