/* 
     File : MetricVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : Abstract base class defining the looping behavior of
            most metric visitors.
             
*/

package mwalk.visitor;


import java.util.Vector;

import mwalk.core.*;
import mwalk.util.MetricCount;
import mwalk.util.VisitException;


/**
 * Abstract base class defining the looping behavior of most metric visitors.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public abstract class MetricVisitor extends PrintVisitor implements DownVisitor {

    
    private static final String DEF_FILENAME = "tree-metric";


    protected long time = 0;
    protected double timeval = 0;
    protected double offset = 0;
    protected long periods = 10;


    public MetricVisitor() {}

    public MetricVisitor( boolean verbose ) {

	super( DEF_FILENAME, verbose );
    }

    public MetricVisitor( long periods ) {

	super( DEF_FILENAME );

	this.periods = periods;
    }

    public MetricVisitor( String filename ) {

	super( filename );
    }

    public MetricVisitor( String filename, long periods, boolean verbose ) {

	super( filename, verbose );

	this.periods = periods;
    }


    public long getTime() {

	return( time );
    }

    public long incTime() {

	timeval += offset;
	return( time = (long)timeval );
    }

    public void setTime( double tv ) {

	timeval = tv;
	time = (long)tv;
    }

    public void startTime( Tree tree ) {

	setTime( (double)tree.getStart() );
    }


    public boolean postfix() {

	return( true );
    }

    public boolean init( Tree tree ) {

	// get the start time for the simulation and compute the incremental offset to use
	startTime( tree );
	if( periods > 1 )
	    // set offset to include enough periods hitting both start and stop times
	    offset = (double)(tree.getStop() - tree.getStart()) / (double)(periods - 1);
	else
	    // set offset to push beyond stop time
	    offset = (double)(tree.getStop() - tree.getStart() + 1);

	Config.verbose( this, "\nComputing metrics at time " + time );

	return( super.init( tree ) );
    }

    public boolean again( Tree tree ) {

	try {
	    process( tree );

	} catch( VisitException ve ) {
	    Config.verbose( ve.getMessage() );
	    return( false );
	}

	// reset for the next pass (if more time)
	reset( tree );
	if( incTime() <= tree.getStop() ) {
	    Config.verbose( this, "\nComputing metrics at time " + time );
	    return( true );
	} else {
	    startTime( tree );
	    return( false );
	}
    }



    public Object visitDown( Tree tree, TreeNode node )  throws VisitException {

	// mark node as seen and visited
	markVisited( node );
	// just return a simple metric count for a single receiver (active or not)
	return( new MetricCount( eval.isActive( (Receiver)node, time ) ) );
    }

    public Object visitDown( Tree tree, TreeNode node, Vector list ) throws VisitException {

	MetricCount count = null;

	// check if this node has been visited yet
	if( ! markVisited( node ) ) {

	    // handle the visit (make it easier to sub-class)
	    count = doVisit( tree, node, list );

	    // save count in source
	    if( node instanceof Source )
		node.data = count;
	}

	// return total count for this sub-tree
	return( count );
    }

    public TreeNode[] getChildren( Tree tree, TreeNode node ) {

	// check if this node has already been seen
	if( ! markSeen( node ) ) {
	    // get all children on active links for the current time
	    return( node.getActiveChildren( eval, time ) );
	} else
	    return( new TreeNode[0] );
    }


    protected abstract MetricCount doVisit( Tree tree, TreeNode node, Vector list ) throws VisitException;

    protected abstract void process( Tree tree ) throws VisitException;
}
