/* 
     File : TraceVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A visitor used by the builder to add mtrace paths into the tree.
             
  $Id: TraceVisitor.java,v 1.3 2000/11/18 04:49:30 robertc Exp $
*/

package mwalk.visitor;


import mwalk.core.*;
import mwalk.util.BuildException;
import mwalk.util.VisitException;


/**
 * A visitor used by the builder to add mtrace paths into the tree.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class TraceVisitor extends AbstractVisitor implements UpVisitor {

    protected MTrace trace;
    TreeNode parent = null;
    protected int hop = 0;


    public TraceVisitor( MTrace trace ) throws BuildException {

	if( trace.parsed )
	    this.trace = trace;
	else
	    throw new BuildException( "visitor was passed unparsed mtrace" );
    }


    public boolean prefix() {

	return( true );
    }


    public void reset( Tree tree ) {

	super.reset( tree );
	// clear parent reference and hop count
	parent = null;
	hop = 0;
    }


    public Object visitUp( Tree tree, TreeNode node ) throws VisitException {

	TreeNode upHop = null;

	// walking will halt if parent is null
	boolean halt = true;
	parent = null;
 
	if( node instanceof Receiver ) {
	    // stop if receiver is not the first hop
	    if( hop > 0 )
		Config.verbose( "TraceVisitor: hit receiver after first hop" );
	    else 
		halt = false;

	} else if( node instanceof Source ) {
	    // stop at source
	    if( hop != trace.hops.length - 1 )
		Config.verbose( "TraceVisitor: hit source before last hop" );

	} else if( hop == 0 ) {
	    // stop if we're starting without a receiver
	    Config.verbose( "TraceVisitor: not starting at receiver" );

	} else if( hop == trace.hops.length - 1 ) {
	    // stop if we're at the end of our hops
	    if( trace.complete )
		Config.verbose( "TraceVisitor: end of complete trace not at source" );

	} else if( trace.hops[hop+1].ip.length() == 0 ) {
	    // we've got an empty hop, just skip it?
	    Config.verbose( "TraceVisitor: skipping empty hop" );
	    hop++;
	    visitUp( tree, node );

	} else if( trace.hops[hop+1].ip.equals( node.getIP() ) ) {
	    // we got a repeat hop, skip to it
	    Config.verbose( "TraceVisitor: skipping duplicate hop" );
	    hop++;
	    visitUp( tree, node );

	} else
	    halt = false;

	if( ! halt ) {
	    // add a new node for the next hop to the tree (if not already there)
	    upHop = tree.addNode( new Router( trace.hops[hop+1].ip ) );
	    // check to make sure that we're not trying to hop to another receiver
	    if( upHop instanceof Receiver ) {
		Config.verbose( "TraceVisitor: hit receiver after first hop" );
		halt = true;
	    }
	}
	
	if( ! halt ) {
	    try {
		// add next hop as this nodes parent router (if not already there)
		parent = node.addParent( upHop );
		TreeNode parents[] = node.getParents();
		long next = Long.MAX_VALUE;
		
		// loop through all other parents and deactivate their link for this time
		for( int p = 0; p < parents.length; p++ )
		    if( ! parents[p].equals( parent ) )
			next = Math.min( next, parents[p].deactivateLink( node, trace.time ) );
		
		// activate the next hop link for this time until the next known trigger time
		parent.activateLink( node, trace.time, next );
		
		// increment the hop count
		hop++;
		
	    } catch( BuildException be ) {
		throw new VisitException( "TraceVisitor error visiting intermediate node", be );
	    } 
	}
	
	return( null );
    }


    public TreeNode[] getParents( Tree tree, TreeNode node ) {

	// return the next hop node or empty to stop walking
	if( parent != null )
	    return( new TreeNode[] { parent } );
	else
	    return( new TreeNode[0] );
    }
}
