#
#     File : TraceSet.pm
#   Author : Robert Chalmers
#
# Original : November 22, 1999
#  Revised :
#
#  Content : class module encapsulating a series of mutlicast/unicast trace attempts
#

package Mwalk::TraceSet;


# import internal classes/modules
use Mwalk::Args;
use Mwalk::MTrace;
use Mwalk::TraceRoute;


# enforce strictness
use strict qw( vars refs subs );

# constant defining the base name of temporary log files used when generating traces
my $TMP_LOG = "trace.tmp.";

# member variable structure
my %members = ( long     => 0,
		len      => 0,
		complete => 0,
		set      => undef,
		args     => undef
	      );



#
# constructor
#
# params
#   class name or object reference
#   argument object
#   optional first mtrace object
#
# return
#   new object reference
#
sub new {

  my ($that, $args, $mtrace) = @_;
  my $class = ref( $that ) || $that;

  # build new object from template
  my $self = { %members };
  bless $self, $class;

  # assign argument object
  $self->{args} = $args;
  # initialize anonymous arrays
  $self->{set} = [];

  # if initial mtrace passed, add it
  $self->add( $mtrace ) if ref( $mtrace );

  return $self;
}



#
# try to add a new mtrace to the set
# only accept it if it's from the same series of attempts
#   1. list is empty
#   2. mtrace is not normal and matches the reciver of prior traces
#
# params
#   self reference
#   reference to a new mtrace
#
# return
#   whether the trace could be added to the list
#
sub add {

  my ($self, $mtrace) = @_;

  # ensure a valid trace was passed in
  ref( $mtrace ) or die "Invalid mtrace object";

  # either the set is empty or the trace is a further attempt
  if( ! @{$self->{set}} or 
      $mtrace->type ne "normal" and $mtrace->{receiver} eq $self->{set}[0]{receiver} ) {
    # check if this has a longer hop count than previous traces (if we're complete forget it)
    #   plus we need to ensure that it starts at the receiver (a -s trace may not)
    if( ! $self->{complete} and 
	 ($mtrace->{complete} or 
        @{$mtrace->{hops}} > $self->{len} and $mtrace->{hops}[0]{ip} eq $mtrace->{receiver}) ) {
      $self->{long} = @{$self->{set}} ;
      $self->{len} = @{$mtrace->{hops}};
      $self->{complete} = $mtrace->{complete};
    }
    return push @{$self->{set}}, $mtrace;
  }

  # didn't fit the bill
  return 0;
}


#
# merge the mtrace set to form the best single trace
# satisfy the constraint that the final mtrace must start at the reciever or return nothing
#
# params
#   self reference
#
# return
#   reference to merged mtrace or undefined if trace is not suitable
#
sub merge {

  my $self = shift;

  # ensure that the list is not empty
  if( @{$self->{set}} > 0 ) {
    # merge the traces into a single representative based off the longest trace
    my $trace = $self->mergeTraces();
    # return that representative if it's valid
    return $trace unless $trace->isTrouble();
  }

  # return undefined otherwise
  return undef;
}


#
# merge the mtrace set to form the best single trace
#
# params
#   self reference
#
# return
#   reference to merged mtrace
#
sub mergeTraces {

  my $self = shift;
  my $len = @{$self->{set}};

  # if there's only one, that's it
  return $self->{set}[0] if $len == 1;

  # try to fill in missing portions from shorter traces
  for( my $i = 0; $i < $len; $i++ ) {
    $self->mergeTrace( $i ) if $i != $self->{long};
  }
  # check if longest is now complete
  $self->{set}[$self->{long}]->isComplete();

  # return the longest trace with holes filled
  return $self->{set}[$self->{long}];
}


#
# merge any useful data from a shorter trace into the longest trace
#
# params
#   self reference
#   index to trace to use to fill in longest
#
sub mergeTrace {

  my ($self, $index) = @_;
  my ($long, $short, $l, $s) = ($self->{set}[$self->{long}], $self->{set}[$index], 0, 0);

  # continue until we've exhausted both lists
  while( $s < @{$short->{hops}} ) {
    # if the short hop is undefined, we should stop
    last unless defined( $short->{hops}[$s]{ip} ); 

    # if we've exhausted the long trace, append the hops from the shorter
    push @{$long->{hops}}, $short->{hops}[$s++] and ++$long->{patch} and next if $l >= @{$long->{hops}};

    # what if the two hops are not matching
    if( ! defined( $long->{hops}[$l]{ip} ) or $long->{hops}[$l]{ip} ne $short->{hops}[$s]{ip} ) {
      # if the shorter trace is a source trace, we need to find where the two meet (if at all)
      next if $short->{type} eq "source" and ! $s and ++$l < @{$long->{hops}};
      # otherwise we're out of luck
      last;
    }

    # we match, so try filling in any missing info
    $long->{hops}[$l]{name} = $short->{hops}[$s]{name} unless $long->{hops}[$l]{name};
    $long->{hops}[$l]{incoming} = $short->{hops}[$s]{incoming} unless $long->{hops}[$l]{incoming};
    if( $long->{hops}[$l]{lost} =~ m/\?/o ) {
      $long->{hops}[$l]{overall} = $short->{hops}[$s]{overall};
      $long->{hops}[$l]{lost} = $short->{hops}[$s]{lost};
      $long->{hops}[$l]{sent} = $short->{hops}[$s]{sent};
      $long->{hops}[$l]{rate} = $short->{hops}[$s]{rate};
    }

    # increment both indices
    $l++; $s++;
  }
}



#
# attempt to generate an mtrace or traceroute  entry
#
# params
#   self reference
#   ip address of receiver
#   type of trace to perform (mtrace, utrace)
#
# return
#   a valid mtrace, traceroute, or undefined
#
sub generate {

  my ($self, $receiver, $type) = @_;

  if( $type eq "mtrace" ) {
    # attempt a source mtrace to identify gateway of receiver
    my $source = $self->generateSource( $receiver );
    if( defined( $source ) ) {
      # attempt a gateway mtrace using the identified gateway
      my $gateway = $self->generateGateway( $source );
      if( defined( $gateway ) ) {
	# add both traces to the set
	$self->add( $gateway );
	$self->add( $source );
	
	# log the traces if requested
	$self->combineLogs( $source, $gateway );
	
	# merge the set and return
	return $self->merge();
      }
    }
  } elsif( $type eq "utrace" ) {
    # attempt to generate a unicast traceroute
    my $utrace = $self->generateUTrace( $receiver );

    if( defined( $utrace ) ) {
      # print the log and return the trace
      $self->printLog( $self->{args}{utraceHandle}, $utrace );
      return $utrace;
    }
  }

  return undef;
}


#
# attempt to generate a source mtrace entry to extract the gateway
#
# params
#   self reference
#   ip address of receiver 
#
# return
#   valid mtrace or undefined
#
sub generateSource {

  my ($self, $receiver) = @_;
  my $mtrace = new Mwalk::MTrace( $self->{args}{strict} );

  # set the source and receiver
  $mtrace->source( $self->{args}{source} );
  $mtrace->receiver( $receiver );

  # launch an mtrace and parse results
  $self->generateMTrace( $mtrace );

  # check if trace was complete
  if( $mtrace->complete() ) {
    return $mtrace;
  } else {
    return undef;
  }
}


#
# attempt to generate a gateway mtrace entry
#
# params
#   self reference
#   complete source mtrace 
#
# return
#   valid mtrace or undefined
#
sub generateGateway {

  my ($self, $source) = @_;
  my $mtrace = new MTrace( $self->{args}{strict} );

  # set the source and receiver and gateway
  $mtrace->source( $source->source() );
  $mtrace->receiver( $source->receiver() );
  $mtrace->gateway( $source->hops()->[1]{ip} );

  # launch an mtrace and parse results
  $self->generateMTrace( $mtrace );

  # check if trace was completely parsed
  if( $mtrace->state() > 0 ) {
    return $mtrace;
  } else {
    return undef;
  }
}


#
# run an mtrace and pipe the output to an mtrace object for pasring
#
# params
#   self reference
#   mtrace to use
#   
# return
#   mtrace after parsing
#
sub generateMTrace {

  my ($self, $mtrace) = @_;
  
  # set mtrace state
  $mtrace->start( $mtrace->stop( $self->{args}{timestamp} ) );
  if( $mtrace->gateway() ) {
    $mtrace->type( "gateway" );
    $mtrace->command( "mtrace -ng " . $mtrace->gateway() . " " . $mtrace->source() . " " . $mtrace->receiver() );
  } else {
    $mtrace->type( "source" );
    $mtrace->flip( 1 );
    $mtrace->command( "mtrace -ns " . $mtrace->receiver() . " " . $mtrace->source() );
  }
  
  # run the trace command and parse the output
  return $self->generateTrace( $mtrace );
}


#
# run a traceroute and pipe the output to a traceroute object for pasring
#
# params
#   self reference
#   receiver ip
#   
# return
#   traceroute after parsing or undefined if not complete
#
sub generateUTrace {

  my ($self, $receiver) = @_;
  my $utrace = new Mwalk::TraceRoute( $self->{args}{strict} );
  
  # set traceroute state
  $utrace->start( $utrace->stop( $self->{args}{timestamp} ) );
  $utrace->receiver( $receiver );
  $utrace->command( "traceroute -n " . $utrace->receiver() );

  # run the trace command and parse the output
  $self->generateTrace( $utrace );
  # return the trace only if it's complete
  return (($utrace->isComplete()) ? $utrace : undef);
}


#
# run a trace command and pipe the output to a traceroute or mtrace object for pasring
#
# params
#   self reference
#   mtrace or traceroute object
#   
# return
#   parsed mtrace or traceroute object
#
sub generateTrace {

  my ($self, $trace) = @_;

  autoflush STDOUT 1;
  print "\ttrace: ", $trace->command() if $self->{args}{verbose};
  autoflush STDOUT 0;

  # launch trace and get pipe on output
  my ($failure, $tmplog) = ("", "$self->{args}{dir}$TMP_LOG" . $trace->type()) ;
  my $pipe = new FileHandle( $trace->command() . " 2>&1 | tee $tmplog |" ) or $failure = "launch: $!";

  unless( $failure ) {
    # parse the trace from the pipe
    $trace->parseLive( $pipe );
    # check trace return value for failure
    close $pipe or $failure = "run: $?";
  }

  print " (failed $failure)" if $failure and $self->{args}{verbose};
  print "\n" if $self->{args}{verbose};
  
  return $trace;
}


#
# combine the temporary logs into a single log file
# the log file is formatted like the MHealth output so it can
# be reread with the same operations.
#
# params
#   self reference
#   reference to a source mtrace
#   reference to a gateway mtrace
#
sub combineLogs {

  my ($self, $source, $gateway) = @_;

  # only do it if logging is enable
  if( $self->{args}{genOutput} ) {
    # ouput the gateway trace, then the source trace
    $self->printLog( $self->{args}{mtraceHandle}, $gateway );
    $self->printLog( $self->{args}{mtraceHandle}, $source );
  }
}


#
# print a temporary log into a single log file
#
# params
#   self reference
#   open filehandle to main log
#   reference to trace
#
sub printLog {

  my ($self, $log, $trace) = @_;
  # open the temporay log file
  my $tmp = new FileHandle( "< $self->{args}{dir}$TMP_LOG" . $trace->type() ) or return;

  # print the start time
  print $log "\n", "#" x 60, "\n# TIMESTAMP START = $self->{args}{timestamp}\n\n";

  # print the command-line
  print $log $trace->command(), "\n";
  # copy temp log into main log
  print $log (<$tmp>);

  # print the stop time
  print $log "\n# TIMESTAMP FINISH = $self->{args}{timestamp}\n";

  close $tmp;
}


#
# clear any exisiting temporary trace logs
#
# params
#   self reference
#
# return
#   number of files cleared
#
sub clearLogs {

  my $self = shift;

  # remove all temporary logs
  return unlink glob( "$self->{args}{dir}$TMP_LOG*" );
}


#
# class destructor
#
# params
#   self reference
#
sub DESTROY {

  my $self = shift;

  #print "Destroying MTrace set.\n";
}



#
# provide access to member variables as automatic methods
#
# params
#   self reference
#
sub AUTOLOAD {

  my $self = shift;
  my $type = ref( $self ) || die "Access method called without object reference: $!";
  my $name = $Mwalk::TraceSet::AUTOLOAD;

  # strip fully-qualified protion of method
  $name =~ s/.*://;
  # ensure member exists
  exists $self->{$name} or die "Access method called on non-existent member: $name";

  # set or get member variable
  if( @_ ) {
    return $self->{$name} = shift;
  } else {
    return $self->{$name};
  }
}
