/* 
     File : AbstractVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : November 17, 2000
            1. Added support for reading config file values in number of 
               formats.

  Content : Abstract class which defines a set of default methods for most visitors.
             
  $Id: AbstractVisitor.java,v 1.5 2000/11/18 04:49:29 robertc Exp $
*/

package mwalk.visitor;


import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;

import mwalk.core.*;
import mwalk.eval.ActivityEvaluator;
import mwalk.util.VisitException;


/**
 * Abstract class which defines a set of default methods for most visitors.
 * This class defines methods for Visitor, UpVisitor, and DownVisitor
 * although it only implements Visitor.  This allows sub-classes to regulate
 * which calls actually get made.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public abstract class AbstractVisitor implements Visitor {

    /** Evaluator used to determine activity when walking */
    protected ActivityEvaluator eval = new ActivityEvaluator();

    /** Hash table used to track who we have seen and processed */
    protected Hashtable seen = new Hashtable();
    /** Whether the visitor prints verbose messages in verbose mode */
    protected boolean verbose = false;


    /**
     * Default constructor.
     */
    public AbstractVisitor() {}


    /**
     * Constructor.
     *
     * @param <code>boolean</code> whether to print verbose messages
     */
    public AbstractVisitor( boolean verbose ) {

	this.verbose = verbose;
    }

    
    /**
     * Return the current activity evaluator.
     *
     * @return <code>ActivityEvaluator</code> current evaluator
     */
    public ActivityEvaluator evaluator() {
	
	return( eval );
    }

    /**
     * Set the current activity evaluator.
     *
     * @param <code>ActivityEvaluator</code> new evaluator
     */
    public void evaluator( ActivityEvaluator eval ) {

	if( eval != null )
	    this.eval = eval;
    }


    /**
     * Check whether the visitor is verbose.
     *
     * @return <code>boolean</code> whether visitor is verbose
     */
    public boolean verbose() {

	return( verbose );
    }


    /**
     * Whether to visit prior to visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit prior
     */
    public boolean prefix() {
	
	return( false );
    }

    /**
     * Whether to visit after visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit after
     */   
    public boolean postfix() {

	return( false );
    }


    /**
     * Initialize the visitor state.
     *
     * @param <code>Tree</code> current tree instance
     */
    public boolean init( Tree tree ) {

	reset( tree );

	return( true );
    }

    /**
     * Reset the visitor for a new pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     */
    public void reset( Tree tree ) {
	
	// clear the list of nodes seen/visited
	seen.clear();
    }

    /**
     * Check whether the visitor wants another pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     * @return <code>boolean</code> whether to walk again
     */
    public boolean again( Tree tree ) {

	return( false );
    }


    /* UpVisitor Methods */

    /**
     * Whether to walk up the tree.
     *
     * @return <code>boolean</code> whether to walk up the tree
     */
    public boolean walkUp() {

	return( true );
    }


    /**
     * Get a list of active receivers.
     *
     * @param <code>Tree</code> current tree instance
     * @return <code>Enumeration</code> list of receivers to visit
     */
    public Enumeration getReceivers( Tree tree ) {

	return( tree.getReceivers() );
    }


    /**
     * Check whether the next receiver should start a visiting path.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>Receiver</code> next receiver
     * @return <code>boolean</code> whether to visit next receiver
     */
    public boolean nextReceiver( Tree tree, Receiver recv ) {

	return( true );
    }


    /**
     * Visit the current node prior to visiting parent.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitUp( Tree tree, TreeNode node ) throws VisitException {

	return( null );
    }


    /**
     * Visit the current node after visiting parent.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @param <code>Object</code> list of return values passed back along the path
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitUp( Tree tree, TreeNode node, Vector list ) throws VisitException {

	return( null );
    }


    /**
     * Retrieve the list of parents of this node.
     *
     * @param <code>Tree</code> vistited tree 
     * @param <code>TreeNode</code> currently visited node
     * @return <code>TreeNode[]</code> array of parent nodes of currently visited node
     */
    public TreeNode[] getParents( Tree tree, TreeNode node ) {

	if( ! markSeen( node ) )
	    return( node.getParents() );
	else
	    return( new TreeNode[0] );
    }


    /* DownVisitor Methods */

    /**
     * Whether to walk down the tree.
     *
     * @return <code>boolean</code> whether to walk down the tree
     */
    public boolean walkDown() {

	return( true );
    }


    /**
     * Visit the current node prior to visiting children.
     * This method is called if prefix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node ) throws VisitException {

	return( null );
    }


    /**
     * Visit the current node after visiting children.
     * This method is called if postfix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @param <code>Object</code> list of return values passed back along the path
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node, Vector list ) throws VisitException {

	return( null );
    }


    /**
     * Retrieve the list of children of this node.
     *
     * @param <code>Tree</code> vistited tree 
     * @param <code>TreeNode</code> currently visited node
     * @return <code>TreeNode[]</code> array of child nodes of currently visited node
     */
    public TreeNode[] getChildren( Tree tree, TreeNode node ) {

	if( ! markSeen( node ) )
	    return( node.getChildren() );
	else
	    return( new TreeNode[0] );
    }


    // node tracking methods

    public boolean haveSeen( TreeNode node ) {

	return( seen.containsKey( node.getIP() ) );
    }

    public boolean haveVisited( TreeNode node ) {

	Object mark = seen.get( node.getIP() );
	return( mark != null && mark instanceof Boolean && ((Boolean)mark).booleanValue() );
    }

    public boolean markSeen( TreeNode node ) {

	if( haveSeen( node ) )
	    return( true );

	// mark as seen but not visited
	seen.put( node.getIP(), Boolean.FALSE );
	return( false );
    }

    public boolean markVisited( TreeNode node ) {

	if( haveVisited( node ) )
	    return( true );

	// mark as seen and visited
	seen.put( node.getIP(), Boolean.TRUE );
	return( false );
    }


    /**
     * Get a configuration value as a string.
     * Returns null if value not found.
     *
     * @param <code>String</code> key string
     * @return <code>String</code> config value
     */
    protected String cfgString( String key ) {

	String str = Config.getCfg( key );

	// return an empty string as null
	if( str != null )
	    if( str.length() == 0 )
		str = null;

	return( str );
    }


    /**
     * Get a configuration value as an integer.
     * Returns -1 if value not found.
     *
     * @param <code>String</code> key string
     * @return <code>int</code> config value
     */
    protected int cfgInt( String key ) {

	String str = cfgString( key );
	int val = -1;

	try {
	    if( str != null )
		// try to convert
		val = Integer.parseInt( str );

	} catch( NumberFormatException nfe ) {}

	return( val );
    }


    /**
     * Get a configuration value as a boolean.
     * Returns false if value not found.
     *
     * @param <code>String</code> key string
     * @return <code>boolean</code> config value
     */
    protected boolean cfgBoolean( String key ) {

	String str = cfgString( key );
	boolean val = false;

	if( str != null ) {
	    // force to lower case
	    str.toLowerCase();
	    // check for positive
	    if( str.equals( "yes" ) || str.equals( "true" ) )
		val = true;
	}   
	return( val );
    }
    

    /**
     * Get a configuration value as a visitor object.
     * Returns null if value not found.
     *
     * @param <code>String</code> key string
     * @return <code>Visitor</code> instantiated visitor
     */
    protected Visitor cfgVisitor( String key ) {

	String str = cfgString( key );
	Visitor visitor = null;

	if( str != null ) 
	    try {
		try {
		    // try to instantiate with passed name
		    visitor = (Visitor)Class.forName( str ).newInstance();
		    
		} catch( ClassNotFoundException cne ) {
		    // try with standard visitor base package
		    visitor = (Visitor)Class.forName( "mwalk.visitor." + str ).newInstance();
		}
	    } catch( Exception e ) {}
	
	return( visitor );
    }
}
