/* 
     File : VisualVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A visitor that outputs a data set that can be used 
            to visualize the tree.
             
  $Id: VisualVisitor.java,v 1.4 2000/11/18 04:49:30 robertc Exp $
*/

package mwalk.visitor;


import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.io.*;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Source;
import mwalk.core.Config;


/**
 * A visitor that outputs a data set that can be used
 * to visualize the tree.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class VisualVisitor extends PrintVisitor implements DownVisitor {

    protected long times[] = null;
    protected int time = 0;
    protected boolean showIP = false;

    public VisualVisitor() {
	super( "tree-visual" );
	times = new long[] { 0 };
	evaluator( new mwalk.eval.RandomEvaluator() );
    }

    public VisualVisitor( long times[] ) {

	this.times = times;
    }


    public boolean prefix() {

	return( true );
    }


    public boolean init( Tree tree ) {

	if( super.init( tree ) ) {
	    if( times == null ) {
		Vector timeList = new Vector( 3, 3 );
		try {
		    BufferedReader in = new BufferedReader( new InputStreamReader( new FileInputStream( "times" ) ) );
		    String line;
		    
		    // parse the list of times to visit
		    while( (line = in.readLine()) != null )
			timeList.add( new Long( line ) );
		    
		} catch( Exception e ) {}
		
		try {
		    // convert list to an array
		    times = new long[timeList.size()];
		    for( int i = 0; i < times.length; i++ )
			times[i] = ((Long)timeList.get( i )).longValue();
		
		} catch( Exception e ) {
		    Config.verbose( "VisualVisitor: error filling time list from file\n\t" + e.getMessage() );
		    return( false );
		}
	    }

	    mwalk.util.ReceiverRandomizer rand = new  mwalk.util.ReceiverRandomizer( tree.receivers() );
	    rand.randomize( tree );

	    return( times.length > 0 );
	} else
	    return( false );
    }

    public void reset( Tree tree ) {

	super.reset( tree );
    }


    public boolean again( Tree tree ) {

	return( super.again( tree ) && ++time < times.length );
    }



    public Object visitDown( Tree tree, TreeNode node ) {

	// mark node as seen and visited
	if( ! haveSeen( node ) ) {
	    StringBuffer buf = new StringBuffer( 50 );

	    buf.append( "NODE [\n\tNAME IP_" + node.getIP() + "\n" );
	    
	    TreeNode[] kids = node.getActiveChildren( eval, times[time] );
	    if( kids.length > 0 ) {
		buf.append( "\tCHILDREN [\n" );
		for( int k =0; k < kids.length; k++ ) 
		    buf.append( "\t\tIP_" + kids[k].getIP() + "\n" );
		buf.append( "\t]\n" );
	    }
	    
	    buf.append( "]\n" );
	    printData( tree, buf.toString() );
	}
	
	return( null );
    }


    public TreeNode[] getChildren( Tree tree, TreeNode node ) {

	// check if this node has already been seen
	if( ! markSeen( node ) )
	    // get all children on active links for the current time
	    return( node.getActiveChildren( eval, times[time] ) );
	else
	    return( new TreeNode[0] );
    }
}
