#
#     File : Args.pm
#   Author : Robert Chalmers
#
# Original : November 22, 1999
#  Revised :
#
#  Content : class module encapsulating global arguments
#

package Mwalk::Args;


# import external classes/modules
use Getopt::Long;
use Sys::Hostname;
use Socket;


# enforce strictness
use strict qw( vars refs subs );

# constant defining the amount to advance timestamp when generating blocks of receivers
my $TIME_ADVANCE = 1000;  # 1 sec.

# member structure
my %members = ( generate     => 0,
	        rtcpFile     => "",
		rtcpHandle   => undef,
		mtraceFile   => "",
                mtraceHandle => undef,
		utraceFile   => "",
                utraceHandle => undef,
		recvFile     => "",
		statFile     => "stats",
	        dir          => ".",
		source       => "",
		block        => 0,
		blocksize    => 10,
		timestamp    => time,
	        timeout      => 240,
	        rtcpOutput   => 0,
	        mtraceOutput => 0,
	        utraceOutput => 0,
		genOutput    => 0,
		append       => 0,
		build        => 0,
		strict       => 0,
		interactive  => 0,
		verbose      => 0,
		start        => 0,
		stop         => 0
	      );


#
# constructor
#
# params
#   class name or object reference
#   whether argument list is for generation
#
# return
#   new object reference
#
sub new {

  my ($that, $gen) = @_ ;
  my $class = ref( $that ) || $that;

  # build new object from template
  my $self = { %members };
  bless $self, $class;

  # set generation flag
  $self->{generation}++ if $gen;

  return $self;
}


#
# set the start time for processing
#
# params
#   self reference
#   start time
#
# return
#   start time
#
sub setStart {

  my ($self, $time) = @_;

  # default to current time
  $time = time unless defined( $time );

  print "start:", scalar localtime $time, "\n\n" if $self->{verbose};

  return $self->{start} = $time;
}


#
# set the stop time for processing
#
# params
#   self reference
#   stop time
#
# return
#   stop time
#
sub setStop {

  my ($self, $time) = @_;

  # default to current time
  $time = time unless defined( $time );

  print "\nstop:", scalar localtime $time, "\nelapsed: ", $time - $self->{start}, " sec.\n\n" if $self->{verbose};

  return $self->{stop} = $time;
}


#
# advance the timestamp by some constant amount when the number
# of receivers fill the current block
#
# params
#   self reference
#   number of receivers
#
sub advanceTime {

  my ($self, $count) = @_;
  # figure which block we're in
  my $block = int $count / $self->{blocksize};

  # check if it's beyond the last block known
  if( $block > $self->{block} ) {
    # increment the timestamp and save the new block
    $self->{timestamp} += $TIME_ADVANCE;
    $self->{block} = $block;
  }
}


#
# parse the command line options
#
# params
#   self reference
#   array of option strings
#
sub parse {

  my ($self, @options) = @_;
  my (%optctl, $ok);

  # process the command line options
  $ok = GetOptions( \%optctl, @options );

  # check for help option
  main::printUsage( 0 ) if exists( $optctl{help} ) and $optctl{help};

  # check for unprocessed arguments
  main::printUsage() if @ARGV or ! $ok;

  # assign options to args hash
  $self->{interactive} = $optctl{interactive} if exists( $optctl{interactive} );
  $self->{verbose} = $optctl{verbose} if exists( $optctl{verbose} );
  $self->{append} = $optctl{append} if exists( $optctl{append} );
  $self->{build} = $optctl{build} if exists( $optctl{build} );
  $self->{strict} = $optctl{fussy} if exists( $optctl{fussy} );
  $self->{rtcpFile} = $optctl{rtcp} if exists( $optctl{rtcp} ) and $optctl{rtcp};
  $self->{mtraceFile} = $optctl{mtrace} if exists( $optctl{mtrace} ) and $optctl{mtrace};
  $self->{utraceFile} = $optctl{utrace} if exists( $optctl{utrace} ) and $optctl{utrace};
  $self->{recvFile} = $optctl{receiver} if exists( $optctl{receiver} ) and $optctl{receiver};
  $self->{statFile} = $optctl{stats} if exists( $optctl{stats} ) and $optctl{stats};
  $self->{dir} = $optctl{dir} if exists( $optctl{dir} ) and $optctl{dir};
  $self->{rtcpOutput} = $optctl{output_rtcp} if exists( $optctl{output_rtcp} );
  $self->{mtraceOutput} = $optctl{output_mtrace} if exists( $optctl{output_mtrace} );
  $self->{utraceOutput} = $optctl{output_utrace} if exists( $optctl{output_utrace} );
  $self->{genOutput} = $optctl{output_gen} if exists( $optctl{output_gen} );
  $self->{source} = $optctl{source} if exists( $optctl{source} );
  $self->{blocksize} = $optctl{block_size} if exists( $optctl{block_size} );
  $self->{timestamp} = $optctl{timestamp} if exists( $optctl{timestamp} );
  $self->{timeout} = $optctl{timeout} if exists( $optctl{timeout} );

  # ensure output directory exists, then append final slash
  -d $self->{dir} or die "Output directory doesn't exist: $self->{dir}\n";
  $self->{dir} .= "/" if $self->{dir} !~ m/\/$/;

  # assign default trace and rtcp logging files
  if( $self->{genOutput} ) {
    $self->{rtcpFile} = "rtcp.log" unless $self->{rtcpFile};
    $self->{rtcpFile} = $self->{dir} . $self->{rtcpFile} unless $self->{rtcpFile} =~ m/\//;
    $self->{mtraceFile} = "mtrace.log" unless $self->{mtraceFile};
    $self->{mtraceFile} = $self->{dir} . $self->{mtraceFile} unless $self->{mtraceFile} =~ m/\//;
    $self->{utraceFile} = "traceroute.log" unless $self->{utraceFile};
    $self->{utraceFile} = $self->{dir} . $self->{utraceFile} unless $self->{utraceFile} =~ m/\//;
  }

  # assign default receiever file
  $self->{recvFile} = $self->{dir} . "receivers" unless $self->{recvFile};

  # prepend out directory to stat file if specified without path
  $self->{statFile} = $self->{dir} . $self->{statFile} if $self->{statFile} and $self->{statFile} !~ m/\//;

  # default source to local host, then convert to an address not host name
  $self->{source} = hostname() unless $self->{source};
  $self->{source} = inet_ntoa( inet_aton( $self->{source} ) );

  # convert seconds to milliseconds for timeout
  $self->{timeout} *= 1000;

  # if verbose, print argument status
  $self->save() if $self->{verbose};
}


#
# print the command-line arguments used
#
# params
#   self reference
#   optional filehandle to use (default STDOUT)
#
sub save {
  
  my ($self, $fh) = @_;

  # default filehandle to stdout
  $fh = \*STDOUT unless ref( $fh );

  # build a list of flags
  my $flags = "";
  foreach my $flag ("interactive", "verbose", "append", "build", "strict") {
    $flags .= "$flag " if $self->{$flag};
  } 

  # print arguments
  print $fh "ARGS:\n";
  print $fh "flags:$flags\n" if $flags;
  print $fh "dir:$self->{dir}\n";
  my $or = ($self->{rtcpOutput} || $self->{genOutput}) ? "(output)" : "";
  print $fh "rtcp:$self->{rtcpFile} $or\n" if $self->{rtcpFile}; 
  my $om = ($self->{mtraceOutput} || $self->{genOutput}) ? "(output)" : "";
  print $fh "mtrace:$self->{mtraceFile} $om\n" if $self->{mtraceFile};
  my $ou = ($self->{utraceOutput} || $self->{genOutput}) ? "(output)" : "";
  print $fh "utrace:$self->{utraceFile} $ou\n" if $self->{utraceFile};
  print $fh "receivers:$self->{recvFile}\n" if $self->{generation};
  print $fh "stats:$self->{statFile}\n" if $self->{statFile};
  print $fh "source:$self->{source}\n" if $self->{generation};
  print $fh "timestamp:$self->{timestamp}\n" if $self->{generation};
  print $fh "timeout:", int $self->{timeout} / 1000, " sec.\n";
}



#
# class destructor
#
# params
#   self reference
#
sub DESTROY {

  my $self = shift;

  #print "Destroying Args.\n";
}



#
# provide access to member variables as automatic methods
#
# params
#   self reference
#
sub AUTOLOAD {

  my $self = shift;
  my $type = ref( $self ) || die "Access method called without object reference: $!";
  my $name = $Mwalk::Args::AUTOLOAD;

  # strip fully-qualified protion of method
  $name =~ s/.*://;
  # ensure member exists
  exists $self->{$name} or die "Access method called on non-existent member: $name";

  # set or get member variable
  if( @_ ) {
    return $self->{$name} = shift;
  } else {
    return $self->{$name};
  }
}
