#
#     File : RTCP.pm
#   Author : Robert Chalmers
#
# Original : November 22, 1999
#  Revised :
#
#  Content : class module encapsulating an individual RTCP packet
#

package Mwalk::RTCP;


# import external modules
use FileHandle;


# enforce strictness
use strict qw( vars refs subs );

# constant pattern for ip address
my $IP = '(?:\d{1,3}\.){3}\d{1,3}';

# member variable structure
my %members = ( state     => 0,
	        receiver  => "",
		timestamp => 0,
		ssrc      => 0,
		bye       => 0,
		stats     => undef
	      );
# stats array element structure
my %stat = ( loss   => 0, 
	     jitter => 0
	   );



#
# constructor
#
# params
#   class name or object reference
#   optional filehandle to start parsing with
#
# return
#   new object reference
#
sub new {

  my ($that, $fh) = @_;
  my $class = ref( $that ) || $that;

  # build new object from template
  my $self = { %members };
  bless $self, $class;

  # initialize anonymous arrays
  $self->{stats} = [];

  # if filehandle passed, then parse file
  $self->parse( $fh ) if ref( $fh );

  return $self;
}



#
# parse a complete rtcp entry
#
# params
#   self reference
#   reference to an open filehandle 
#
# return
#   whether parse completed successfully (1=success, 0=fail, -1=skip)
#   the return value is also saved in {state}
#
sub parse {

  my ($self, $fh) = @_;
  my $line;

  # ensure a valid file handle was passed
  ref( $fh ) or die "File handle not defined";
  
  # reset state to failure
  $self->{state} = 0;

  while( $line = <$fh> ) {
    # skip comments
    next if $line =~ m/^\s*\#/o;

    # parse out everything other than variable length stats
    if( $line =~ m/(\d+) ($IP) (\d+) (\d+)[\d ]*(B)?/o ) {
      $self->{timestamp} = $1;
      $self->{receiver} = $2;
      $self->{ssrc} = $3;
      $self->{bye} = defined( $5 );

      my ($count, @stats) = ($4, split( " ", $line ));
      # split out variable length stats
      #   convert loss into percentage from fixed point 8-bit binary
      for( my $i = 0; $i < $count; $i++ ) {
	push @{$self->{stats}}, {loss => ($stats[4+$i] / 256) * 100, jitter => $stats[5+$i]};
      }

      # return successful
      return ++$self->{state};
    }

    # return unsuccessful
    return $self->{state};
  }

  # skip since we never got a chance to parse
  return --$self->{state};
}


#
# generate a fake rtcp entry for the given ip at the curren time
#
# params
#   self reference
#   receiver ip
#   argument object
#
sub generate {

  my ($self, $ip, $args) = @_;

  # fill-in rtcp info
  $self->{state} = 1;
  $self->{receiver} = $ip;
  $self->{timestamp} = $args->{timestamp};

  # print an rtcp log entry without an ssrc or stats
  print { $args->{rtcpHandle} } "$self->{timestamp} $self->{receiver} 0 0\n" if $args->{genOutput};
}


#
# write the processed rtcp out to file
#
# params
#   self reference
#   name of file to save to or open file handle
#
sub save {

  my ($self, $file) = @_;
  my $fh;

  # open the save file unless handle was provided
  if( ref( $file ) ) {
    $fh = $file; 
  } else {
    $fh = new FileHandle( ">> $file" ) or die "Can't open save file $file: $!\n";
  }

  # output the rtcp data with labels
  print $fh "RTCP:", ($self->{bye}) ? "B" : "U", "\n";
  print $fh "receiver:$self->{receiver}\nssrc:$self->{ssrc}\n";
  print $fh "time:$self->{timestamp}\n";
  print $fh "stats:", scalar @{$self->{stats}}, "\n";
  # print rtcp stats
  for( my $i = 0; $i < @{$self->{stats}}; $i++ ) {
    # each line of stats is actually a hash
    printf $fh "%3d [%.2f] [%d]\n", $i, $self->{stats}[$i]{loss}, $self->{stats}[$i]{jitter};
  }
  print $fh "\n";

  # close the file if we opened it
  close $fh unless ref( $file );
}



#
# class destructor
#
# params
#   self reference
#
sub DESTROY {

  my $self = shift;

  #print "Destroying RTCP for $self->{receiver}.\n";
}



#
# provide access to member variables as automatic methods
#
# params
#   self reference
#
sub AUTOLOAD {

  my $self = shift;
  my $type = ref( $self ) || die "Access method called without object reference: $!";
  my $name = $Mwalk::RTCP::AUTOLOAD;

  # strip fully-qualified protion of method
  $name =~ s/.*://;
  # ensure member exists
  exists $self->{$name} or die "Access method called on non-existent member: $name";

  # set or get member variable
  if( @_ ) {
    return $self->{$name} = shift;
  } else {
    return $self->{$name};
  }
}
