/* 
     File : Config.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : November 17, 2000
            1. Added configuration file command-line option with load/get
               methods for handling configuration key/value pairs.

  Content : A static class that is used to parse the command-line for
            options and provides some common functionality to all
	    other system components.
             
  $Id: Config.java,v 1.4 2000/11/18 08:52:21 robertc Exp $
*/

package mwalk.core;


import java.util.StringTokenizer;
import java.util.Properties;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.net.InetAddress;

import mwalk.visitor.Visitor;


/**
 * A static class that is used to parse the command-line for options and 
 * provides some common functionality to all other system components.
 * <p>
 * If a configuration file is passed in then that file will be read an its
 * contents will be made available to other components through as key/value
 * pairs.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class Config {

    /** Name of input tree file */
    public static String readTree = null;
    /** Name of output tree file */
    public static String writeTree = null;
    /** List of directories to check for input files */
    public static String dirs[] = new String[0];
    /** List of visitors to animate with */
    public static String visitors[] = new String[0];
    /** Source host or IP address to use in new tree */
    public static String source = null;
    /** Verbosity flag */
    public static boolean verbose = false;
    /** Whether to pause at each node before continuing walk */
    public static boolean pause = false;
    
    /** Configuration file name */
    protected static String cfgFile = null;
    /** Parsed configuration hash table */
    protected static Properties cfgProp = new Properties();
    /** Internal parsing flag */
    protected static boolean skip = false;


    /**
     * Default constructor.
     * Protected to disallow creation of Config objects.  
     * There should only be a single static Config object.
     * used as a static object.
     */
    protected Config() {}


    /**
     * Parse the command-line arguments.
     *
     * @param <code>String[]</code> command-line arguments
     */
    public static void parseArgs( String args[] ) {
	
	try {
	    // handle command-line arguments 
	    for( int arg = 0; arg < args.length; arg++ )
		// check if this argument was used by a preceding flag
		if( skip )
		    skip = false;
		else if( args[arg].startsWith( "-" ) ) {
		    // check each flag value
		    for( int flag = 1; flag < args[arg].length(); flag++ )
			if( ! parseFlag( args, arg, args[arg].charAt( flag ) ) )
			    printUsage();
		} else { 
		    // load configuration file if passed
		    loadCfg( cfgFile );

		    // figure whether final parameter is tree file or source
		    File check = new File( args[ arg ] );
		    if( check.canRead() )
			readTree = args[ arg ];
		    else {
			InetAddress.getByName( args[ arg ] );
			source = args[ arg ];
		    }
		    break;
		}
	    
	} catch( Exception e ) {
	    printUsage();
	}

	// we need to start with a tree or know the source of the tree
	if( readTree == null && source == null )
	    printUsage();
    }
    
    
    /**
     * Set the skip flag if the next argument has been used.
     * If skip is already set, then print usage.
     */
    private static void setSkip() {
	
	// only set skip once per flag set
	if( skip )
	    printUsage();
	else
	    skip = true;
    }

    
    /**
     * Parse parameter flags.
     *
     * @param <code>String[]</code> argument list
     * @param <code>int</code> current argument
     * @param <code>char</code> current flag
     * @return <code>boolean</code> whether the flag was handled
     */
    private static boolean parseFlag( String[] args, int arg, char flag ) {

	switch( flag ) {
	case 'c' :
	    cfgFile = args[arg + 1];
	    setSkip();
	    break;
	case 'o' : 
	    writeTree = args[arg + 1];
	    setSkip();
	    break;
	case 'd' :
	    setDirs( args[arg + 1] );
	    setSkip();
	    break;
	case 'a' :
	    setVisitors( args[arg + 1] );
	    setSkip();
	    break;
	case 'v' : 
	    verbose = true;
	    break;
	case 'p' : 
	    pause = true;
	    break;
	case 'h' :
	case '?' :
	    printUsage();
	    break;
	default: 
	    return( false );
	}

	return( true );
    }


    /**
     * Print usage information.
     */
    public static void printUsage() {

	System.out.println( "Usage: mwalk <action> [-h] [-v] [-p] [-c <cfgFile>] [-d <dirList>]" );
	System.out.println( "             [-a <visitorList>] [-o <outFile>] <source>|<treeFile>" );
	System.out.println( "Actions:" );
	System.out.println( "  build    build an activity graph representing the entire session" );
	System.out.println( "  animate  animate the activity graph with visitors" );
	System.out.println( "Parameters:" );
	System.out.println( "  source    indicate which source to use from parsed logs" );
	System.out.println( "  treeFile  indicate an existing tree to expand or animate" );
	System.out.println( "Options:" );
	System.out.println( "  -h  print this message" );
	System.out.println( "  -v  operate with verbose messages" );
	System.out.println( "  -p  pause before visiting a nodes children/parent" );
	System.out.println( "  -c  path to configuration file" );
	System.out.println( "  -d  list of paths to parsed logs" );
	System.out.println( "  -a  list of visitor classes" );
	System.out.println( "  -o  output file for binary representation" );
	System.out.println();
	System.exit( 0 );
    }


    /**
     * Parse an array of directories from a comma separated list.
     *
     * @param <code>String</code> list of directories
     */
    public static void setDirs( String dirList ) {

	try {
	    StringTokenizer toker = new StringTokenizer( dirList, "," );
	    String slash = System.getProperty( "file.separator" );

	    dirs = new String[ toker.countTokens() ];
	    
	    for( int i = 0; i < dirs.length; i++ ) {
		String tok = toker.nextToken();
		dirs[i] = (tok.endsWith( slash )) ? tok : tok + slash;
	    }
	} catch( Exception e ) {
	    printUsage();
	}
    }


    /**
     * Parse and array of visitor classes from a comma separated list.
     *
     * @param <code>String</code> list of visitors
     */
    public static void setVisitors( String visitorList ) {

	try {
	    StringTokenizer toker = new StringTokenizer( visitorList, "," );

	    visitors = new String[ toker.countTokens() ];
	    
	    for( int i = 0; i < visitors.length; i++ )
		visitors[i] = toker.nextToken();

	} catch( Exception e ) {
	    printUsage();
	}
    }


    /**
     * Load configuration file as a set of key/value pairs.
     *
     * @param <code>String</code> complete path to config file
     */
    public static void loadCfg( String cfgFile ) {

	try {
	    verbose( "Reading configration file: " + cfgFile + "\n" );
	    cfgProp.load( new FileInputStream( cfgFile ) );

	} catch( Exception e ) {
	    System.err.println( "Warning: configuration file could not be read." );
	}
    }


    /**
     * Lookup a configuration value based on its key.
     * 
     * @param <code>String</code> config key
     * @return <code>String</code> config value
     */
    public static String getCfg( String key ) {
	
	return( cfgProp.getProperty( key ) );
    }


    /**
     * Print if the verbose flag is set.
     *
     * @param <code>String</code> text to print
     */
    public static void verbose( String debug ) {

	if( verbose )
	    System.out.println( debug );
    }


    /**
     * Print if the visitor is verbose and the verbose flag is set.
     *
     * @param <code>Visitor</code> visitor to check for verbosity
     * @param <code>String</code> text to print
     */
    public static void verbose( Visitor visitor, String debug ) {

	if( verbose && visitor.verbose() )
	    System.out.println( debug );
    }


    /**
     * Print if the verbose flag is set.
     *
     * @param <code>Exception</code> exception with text to print
     */    
    public static void verbose( Exception debug ) {

	if( verbose )
	    debug.printStackTrace();
    }


    /**
     * Pause and wait for a keystroke if the pause flag is set.
     */
    public static void pause() {

	pause( null, null );
    }

    /**
     * Pause and wait for a keystroke if the pause flag is set.
     *
     * @param <code>String</code> prompt string
     */
    public static void pause( String prompt ) {

	pause( null, prompt );
    }
    
    /**
     * Pause and wait for a keystroke if the pause flag is set and the 
     * visitor is verbose.
     *
     * @param <code>Visitor</code> visitor to check for verbosity
     */
    public static void pause( Visitor visitor ) {
	
	pause( visitor, null );
    }


    /**
     * Pause and wait for a keystroke if the pause flag is set and the 
     * visitor is verbose.
     *
     * @param <code>Visitor</code> visitor to check for verbosity
     * @param <code>String</code> prompt string
     */
    public static void pause( Visitor visitor, String prompt ) {

	if( pause && (visitor == null || visitor.verbose()) )
	    try {
		if( prompt != null )
		    System.out.print( prompt );
		
		BufferedReader br = new BufferedReader( new InputStreamReader( System.in ) );
		br.readLine();
		
	    } catch( Exception e ) {}
    }
}
