/* 
     File : Hop.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A class representing a single hop in an mtrace.
             
  $Id: Hop.java,v 1.3 2000/11/18 04:49:28 robertc Exp $
*/

package mwalk.core;


import mwalk.util.BuildException;


/**
 * A class representing a single hop in an mtrace.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class Hop {

    /** IP address of next hop */
    public String ip;
    /** Hostname of next hop */
    public String name;
    /** Upstream router */
    public String upstream; 
    /** Loss ratio */
    public String  ratio;
    /** Overall number of packets seen */
    public int overall;
    /** Packet rate */
    public int rate;

    /** Parsing state */
    private int open, close;


    /**
     * Constructor.
     *
     * @param <code>String</code> hop line to parse
     * @exception <code>BuildException</code> if hop could not be parsed
     */
    public Hop( String line ) throws BuildException {

	parse( line );
    }

    
    /**
     * Parse a hop line into its constituent parts.
     *
     * @param <code>String</code> hop line to parse
     * @exception <code>BuildException</code> if hop could not be parsed
     */
    public void parse( String line ) throws BuildException {

	// reset parsing state
	open = close = 0;

	// parse out hop info
	ip = nextString( line );
	name = nextString( line );
	upstream = nextString( line );
	overall = nextInt( line );
	ratio = nextString( line );
	rate = nextInt( line );
    }


    /**
     * Parse the next string enclosed by brackets.
     *
     * <code>String</code> line to parse
     * @exception <code>BuildException</code> if string could not be parsed
     */
    protected String nextString( String line ) throws BuildException {

	// hop values are surrounded by brackets
	open = line.indexOf( '[', close );
	close = line.indexOf( ']', open );

	if( open < 0 || close < 0 ) 
	    throw new BuildException( "error parsing hop value" );

	if( close - open == 1 )
	    return( "" );
	else
	    return( line.substring( open + 1, close ) );
    }


    /**
     * Parse the next integer enclosed by brackets.
     *
     * <code>String</code> line to parse
     * @exception <code>BuildException</code> if integer could not be parsed
     */
    protected int nextInt( String line ) throws BuildException {

	try {
	    return( Integer.parseInt( nextString( line ) ) );
	} catch( Exception e ) {
	    throw new BuildException( "error parsing hop int value" );
	}
    }
}
