/* 
     File : DegreeCountVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A visitor used to count the degree of each node along the
            active tree.
             
*/

package mwalk.visitor;


import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;
import mwalk.core.Config;
import mwalk.util.DegreeCount;
import mwalk.util.VisitException;


/**
 * A visitor used to count the degree of each node along the
 * active tree. This visitor assumes that the node's data contain
 * the current depth, otherwise the depth will be reported as zero.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class DegreeCountVisitor extends PrintVisitor implements DownVisitor {

    private static final String DEF_FILENAME = "tree-degree";


    public DegreeCountVisitor() {

	super( DEF_FILENAME );
    }

    public DegreeCountVisitor( boolean verbose ) {

	super( DEF_FILENAME, verbose );
    }

    public DegreeCountVisitor( String filename ) {

	super( filename );
    }

    public DegreeCountVisitor( String filename, boolean verbose ) {

	super( filename, verbose );
    }


    public boolean postfix() {

	return( true );
    }


    protected void printHeader( Tree tree ) {

	super.printHeader( tree );

	ps.println( "#" );
	ps.println( "# depth\tdownstream receivers\ttotal degree\tgateway degree\tnon-gateway degree" );
    }


    public Object visitDown( Tree tree, TreeNode node ) {

	// mark node as seen and visited
	markVisited( node );
	// just return a simple degree count for a single receiver (active or not)
	return( new DegreeCount( eval.isActive( (Receiver)node ) ) );
    }

    public Object visitDown( Tree tree, TreeNode node, Vector list ) throws VisitException {

	DegreeCount kids = new DegreeCount();
	
	// mark node as seen and visited
	if( ! markVisited( node ) ) {
	    // figure whether this node is used in the current active tree
	    for( Enumeration enum = list.elements(); enum.hasMoreElements(); )
		try {
		    // get count from next child
		    DegreeCount count = (DegreeCount)enum.nextElement();
		    // add count to current total
		    kids.add( count );
		    		    
		} catch( NoSuchElementException nse ) {
		} catch( Exception e ) {
		    throw new VisitException( "error evaluating child activity" );
		}

	    if( kids.receivers > 0 ) {
		int depth = 0;
		try {
		    // figure out our current depth
		    depth = ((Integer)node.data).intValue();
		    
		} catch( Exception e ) {}
		
		// print out depth and degree of node
		String line = depth + "\t" + kids.toString() + "\t" + (kids.links - kids.glinks);
		
		Config.verbose( this, line );
		printData( tree, line );
	    }
	}
	
	return( kids );
    }
}
