/* 
     File : UpathMetricVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A basic implementation of a visitor that gathers metric data, but
            uses real unicast path lengths in calculations rather than assuming
	    multicast paths equate to unicast paths.
             
*/

package mwalk.visitor;


import java.util.StringTokenizer;
import java.io.*;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;
import mwalk.core.Config;
import mwalk.eval.RandomEvaluator;
import mwalk.util.MetricCount;
import mwalk.util.VisitException;


/**
 * An implementation of a visitor that gathers metric data, but
 * uses real unicast path lengths in calculations rather than assuming
 * multicast paths equate to unicast paths.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class UpathMetricVisitor extends BasicMetricVisitor {

    private static final String DEF_FILENAME = "tree-metric.upath";

    protected String pathFile = "receivers.path";


    public UpathMetricVisitor() {

	super( DEF_FILENAME );
    }

    public UpathMetricVisitor( boolean verbose ) {

	super( verbose );

	setFilename( DEF_FILENAME );
    }

    public UpathMetricVisitor( String filename ) {

	super( filename );
    }

    public UpathMetricVisitor( long periods ) {

	super( DEF_FILENAME, periods, false );
    }

    public UpathMetricVisitor( String filename, long periods, boolean verbose ) {

	super( filename, periods, verbose );
    }


    public boolean init( Tree tree ) {

	String visitor = getClass().getName();
	try {
	    // open path length file
	    BufferedReader in = new BufferedReader( new InputStreamReader( new FileInputStream( pathFile ) ) );
	    int recvs = 0;

	    while( in.ready() ) {
		String line = in.readLine();
		if( ! line.startsWith( "#" ) ) {
		    // break out line into ip, upath length
		    StringTokenizer tok = new StringTokenizer( line );
		    String ip = tok.nextToken();
		    Integer upath = new Integer( tok.nextToken() );
		    // get the receiver matching the ip
		    Receiver recv = tree.getReceiver( ip );
		    if( recv != null ) {
			// set the unicast path
			recv.data = upath;
			recvs++;
		    }
		}
	    }
	    Config.verbose( this, "Loaded unicast paths for " + recvs + " receivers." );

	} catch( Exception e ) {
	    Config.verbose( visitor + ".init(): error setting unicast path lengths" );
	    return( false );
	}

	return( super.init( tree ) );
    }


    public Object visitDown( Tree tree, TreeNode node ) throws VisitException {

	// mark node as seen and visited
	markVisited( node );
	// return a metric count setting the unicast path length
	MetricCount count = new MetricCount( eval.isActive( (Receiver)node, time ) );
	try {
	    if( count.receivers > 0 )
		count.ucastLinks = ((Integer)node.data).intValue();

	} catch( Exception e ) {
	    // consider receiver inactive if no valid unicast path length could be obtained
	    count = new MetricCount( false );
	}
	
	return( count );
    }

    protected MetricCount addLinks( MetricCount count, int mlinks ) {

	// add ourselves into the pot (with optional optimization for multicast)
	if( count.receivers > 0 ) {
	    Config.verbose( this, "Adding counts for " + count.receivers + " receivers." );
	    count.mcastLinks += (optMulti) ? 1 : mlinks;
	}

	return( count );
    }
}
