/* 
     File : TreeNode.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : An abstract class representing a node in the tree.
             
  $Id: TreeNode.java,v 1.6 2000/11/18 04:49:28 robertc Exp $
*/

package mwalk.core;


import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.io.Serializable;

import mwalk.eval.ActivityEvaluator;
import mwalk.util.BuildException;


/**
 * An abstract class representing a node in the tree.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public abstract class TreeNode implements Serializable {

    /** Serialized version ID */
    static final long serialVersionUID = 2301603752167076822L;

    /** IP address of node */
    protected String ip = null;
    /** List of parent nodes hashed by IP */
    protected Hashtable parents = null;
    /** List of child nodes hashed by IP */
    protected Hashtable children = null;

    /** Flag indicating randomly chosen activity */
    public transient boolean active = false;
    /** Storage available to visitors to store partial results */
    public transient Object data = null;


    /**
     * Constructor.
     *
     * @param <code>String</code> node's IP address
     */
    public TreeNode( String ip ) {

	this.ip = ip.trim();
    }
    

    /**
     * Return the node's IP address.
     *
     * @return <code>String</code> node's IP address
     */
    public String getIP() {

	return( ip );
    }


    /**
     * Check whether this node has a particular parent.
     *
     * @param <code>String</code> address of parent
     * @return <code>boolean</code> whether node has this parent
     */
    public boolean hasParent( String ip ) {

	return( parents != null && parents.containsKey( ip ) );
    }


    /**
     * Return the numbmer of parents.
     *
     * @return <code>int</code> number of parents
     */
    public int parents() {

	if( parents == null )
	    return( 0 );
	else
	    return( parents.size() );
    }

    /** 
     * Return a list of this node's parents.
     *
     * @return <code>TreeNode[]</code> list of parents
     */
    public TreeNode[] getParents() {

	if( parents == null )
	    return( null );
	else
	    return( toNodeArray( parents.elements(), parents.size() ) );
    }


    /**
     * Check whether this node has a particular child.
     *
     * @param <code>String</code> address of child
     * @return <code>boolean</code> whether node has this child
     */
    public boolean hasChild( String ip ) {

	return( children != null && children.containsKey( ip ) );
    }

    /**
     * Return the numbmer of children.
     *
     * @return <code>int</code> number of children
     */
    public int children() {

	if( children == null )
	    return( 0 );
	else
	    return( children.size() );
    }

    /** 
     * Return a list of this node's children.
     *
     * @return <code>TreeNode[]</code> list of children
     */
    public TreeNode[] getChildren() {

	if( children == null )
	    return( null );
	else
	    return( toNodeArray( children.elements(), children.size() ) );
    }


    public Link getChildLink( TreeNode child ) {

	if( children == null )
	    return( null );
	else
	    return( (Link)children.get( child.ip ) );
    }

    public Link[] getChildLinks() {

	if( children == null )
	    return( null );
	else
	    return( toLinkArray( children.elements(), children.size() ) );
    }


    public TreeNode addParent( TreeNode parent ) {

	if( parents == null )
	    return null;

	// return an existing parent with this ip or add it
	if( parents.containsKey( parent.ip ) )
	    parent = (TreeNode)parents.get( parent.ip );
	else
	    parents.put( parent.ip, parent );

	// add this node as a child to the parent
	parent.addChild( this );

	return( parent );
    }

    public TreeNode addChild( TreeNode child ) {

	if( children == null )
	    return( null );
	else if( children.containsKey( child.ip ) )
	    return( ((Link)children.get( child.ip )).node );
	else {
	    children.put( child.ip, new Link( child ) );
	    return( child );
	}	
    }

    protected Link addChildLink( TreeNode child ) {

	if( children == null )
	    return( null );
	else if( children.containsKey( child.ip ) )
	    return( (Link)children.get( child.ip ) );
	else {
	    Link link = new Link( child );
	    children.put( child.ip, link );
	    return( link );
	}	
    }


    public void delParent( TreeNode parent ) {

	if( parents != null )
	    parents.remove( parent.ip );
    }

    public void delChild( TreeNode child ) {

	if( children != null )
	    children.remove( child.ip );
    }


    public void activateLink( TreeNode child, long time, long next ) throws BuildException {

	Link link = addChildLink( child );

	if( link != null )
	    link.activate( time, next );
	else
	    throw new BuildException( "error activating a non-existent link from " + ip + " to " + child.ip );
    }


    public long deactivateLink( TreeNode child, long time ) throws BuildException {

	Link link = addChildLink( child );

	if( link != null )
	    return( link.deactivate( time ) );
	else
	   throw new BuildException( "error deactivating a non-existent link from " + ip + " to " + child.ip ); 
    }


    public TreeNode[] getActiveChildren( ActivityEvaluator eval ) {

	// activity must not be time-dependent
	return( getActiveChildren( eval, 0 ) );
    }

    public TreeNode[] getActiveChildren( ActivityEvaluator eval, long time ) {

	if( children != null ) {
	    Vector links = getActiveList( eval, time );
	    return( toNodeArray( links.elements(), links.size() ) );
	} else
	    return( new TreeNode[0] );
    }


    public Link[] getActiveLinks( ActivityEvaluator eval ) {

	// activity must not be time-dependent
	return( getActiveLinks( eval, 0 ) );
    }

    public Link[] getActiveLinks( ActivityEvaluator eval, long time ) {

	if( children != null ) {
	    Vector links = getActiveList( eval, time );
	    return( toLinkArray( links.elements(), links.size() ) );
	} else
	    return( new Link[0] );
    }

    protected Vector getActiveList( ActivityEvaluator eval, long time ) {

	Vector active = new Vector( children.size() );

	// get a list of links that report activity for this time
	for( Enumeration links = children.elements(); links.hasMoreElements(); )
	    try {
		Link link = (Link)links.nextElement();
		if( eval.isActive( link, time ) )
		    active.add( link );

	    } catch( NoSuchElementException nse ) {}

	return( active );
    }


    public void clearData() {

	if( children != null )
	    // clear data for each associate link
	    for( Enumeration links = children.elements(); links.hasMoreElements(); )
		try {
		    ((Link)links.nextElement()).clearData();
		    
		} catch( Exception e ) {}

	// clear local data
	data = null;
    }


    public boolean equals( TreeNode node ) {

	return( ip.equals( node.ip ) );
    }

    public boolean equals( String check ) {

	return( ip.equals( check.trim() ) );
    }


    public static Vector toNodeVector( Enumeration list, int size ) {

	Vector nodes = new Vector( size, 5 );

	try {
	    for( int n = 0; n < size; n++ ) {
		Object ele = list.nextElement();
		if( ele instanceof Link )
		    nodes.add( ((Link)ele).node );
		else
		    nodes.add( ele );
	    }
	} catch( NoSuchElementException nse ) {}

	return( nodes );
    }


    public static TreeNode[] toNodeArray( Enumeration list, int size ) {

	TreeNode nodes[] = new TreeNode[ size ];

	try {
	    for( int n = 0; n < nodes.length; n++ ) {
		Object ele = list.nextElement();
		if( ele instanceof Link )
		    nodes[n] = ((Link)ele).node;
		else
		    nodes[n] = (TreeNode)ele;
	    }
	} catch( NoSuchElementException nse ) {}

	return( nodes );
    }


    public static Link[] toLinkArray( Enumeration list, int size ) {

	Link links[] = new Link[ size ];

	try {
	    for( int n = 0; n < links.length; n++ )
		links[n] = (Link)list.nextElement();

	} catch( NoSuchElementException nse ) {}

	return( links );
    }
}
