/* 
     File : DegreeCountVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A visitor used to count the degree of each node along the
            active tree.
             
*/

package mwalk.visitor;


import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;
import mwalk.core.Config;
import mwalk.util.DegreeCount;
import mwalk.util.VisitException;


/**
 * A visitor used to count the degree of each node along the
 * active tree. This visitor assumes that the node's data contain
 * the current depth, otherwise the depth will be reported as zero.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class DegreeCountVisitor extends PrintVisitor implements DownVisitor {

    /** Base of keys in configuration file */
    private static final String DEF_FILENAME = "tree-degree";


    /**
     * Default constructor.
     */
    public DegreeCountVisitor() {

	super( DEF_FILENAME );
    }
    
    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     */
    public DegreeCountVisitor( String filename ) {

	super( filename );
    }


    /**
     * Whether to visit after visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit after
     */   
    public boolean postfix() {

	return( true );
    }


    /**
     * Print visitor's header info at start of file.
     *
     * @param <code>Tree</code> current tree instance
     */
    protected void printHeader( Tree tree ) {

	super.printHeader( tree );

	ps.println( "#" );
	ps.println( "# depth\tdownstream receivers\ttotal degree\tgateway degree\tnon-gateway degree" );
    }


    /**
     * Visit the current node prior to visiting children.
     * This method is called if prefix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node ) {

	// mark node as seen and visited
	markVisited( node );
	// just return a simple degree count for a single receiver (active or not)
	return( new DegreeCount( eval.isActive( (Receiver)node ) ) );
    }

    /**
     * Visit the current node after visiting children.
     * This method is called if postfix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @param <code>Vector</code> list of return values passed back along the path
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node, Vector list ) throws VisitException {

	DegreeCount kids = new DegreeCount();
	
	// mark node as seen and visited
	if( ! markVisited( node ) ) {
	    // figure whether this node is used in the current active tree
	    for( Enumeration enum = list.elements(); enum.hasMoreElements(); )
		try {
		    // get count from next child
		    DegreeCount count = (DegreeCount)enum.nextElement();
		    // add count to current total
		    kids.add( count );
		    		    
		} catch( NoSuchElementException nse ) {
		} catch( Exception e ) {
		    throw new VisitException( "error evaluating child activity" );
		}

	    if( kids.receivers > 0 ) {
		int depth = 0;
		try {
		    // figure out our current depth
		    depth = ((Integer)node.data).intValue();
		    
		} catch( Exception e ) {}
		
		// print out depth and degree of node
		String line = depth + "\t" + kids.toString() + "\t" + (kids.links - kids.glinks);
		
		Config.verbose( this, line );
		printData( tree, line );
	    }
	}
	
	return( kids );
    }
}
