/* 
     File : ExportVisitor.java
   Author : Robert Chalmers

 Original : November 17, 2000
  Revised : 

  Content : A visitor that exports an ascii representation of the current tree.
             
  $Id: ExportVisitor.java,v 1.2 2001/08/22 17:45:07 robertc Exp $
*/

package mwalk.visitor;


import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.io.*;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Source;
import mwalk.core.Config;


/**
 * A visitor that exports an ascii representation of the current tree.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class ExportVisitor extends PrintVisitor implements DownVisitor {

    /** Base of keys in configuration file */
    private static final String CFG_BASE = "ExportVisitor.";
    /** Default file name if none supplied */
    private static final String DEF_FILENAME = "tree-export";

    /** Whether to export true IP address or whether to mangle it */
    protected boolean showIP = false;
    /** Integer count used to assign virtual IDs to nodes */
    protected int nextID = 0;


    /**
     * Default constructor.
     */
    public ExportVisitor() {
	
	super( DEF_FILENAME );
    }


    /**
     * Constructor.
     *
     * @param <code>String</code> export filename
     */
    public ExportVisitor( String filename ) {
	
	super( filename );
    }
    

    /*
     * Visit the current node prior to visiting its children.
     *
     * @return <code>boolean</code> visit node first
     */
    public boolean prefix() {

	return( true );
    }


    /**
     * Initialize visitor.
     *
     * @param <code>Tree</code> current tree instance
     */
    public boolean init( Tree tree ) {

	// allow config to override showIP
 	if( cfgBoolean( CFG_BASE + "showIP" ) )
	    showIP = true;

	// setup a random evaluator
	evaluator( new mwalk.eval.RandomEvaluator() );

	return( super.init( tree ) );
    }

    /**
     * Reset visitor state.
     *
     * @param <code>Tree</code> current tree instance
     */
    public void reset( Tree tree ) {

	super.reset( tree );
    }


    /**
     * Determine whether to walk the tree again.
     *
     * @param <code>Tree</code> current tree instance
     * @return <code>boolean</code> whether to walk again
     */
    public boolean again( Tree tree ) {

	return( super.again( tree ) );
    }


    /**
     * Visit the current node prior to visiting children.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node ) {

	// mark node as seen and visited
	if( ! haveSeen( node ) ) {
	    StringBuffer buf = new StringBuffer( 50 );

	    buf.append( getID( node ) );
	    
	    TreeNode[] kids = node.getActiveChildren( eval );
	    for( int k = 0; k < kids.length; k++ ) 
		buf.append( "\t" + getID( kids[k] ) );

	    printData( tree, buf.toString() );

	}
	 	
	return( null );
    }


    /**
     * Return the node's ID.
     * If showIP is not true, then the IP address will be mangled first.
     *
     * @param <code>TreeNode</code> current node
     * @return <code>String</code> node's ID
     */
    protected String getID( TreeNode node ) {

	String id = node.getIP();

	/* mangle IP unless requested otherwise */
	if( ! showIP ) {
	    // check if node already has an ID
	    Integer newID = (Integer)node.data;
	    if( newID == null )
		// create a new ID
		node.data = newID = new Integer( ++nextID );
	    id = newID.toString();
	}

	return( id );
    }
}
