/* 
     File : JoinVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A visitor used to mark the initial join time along a receiver's
            earliest path.
             
*/

package mwalk.visitor;


import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;
import mwalk.core.Link;


/**
 * A visitor used to mark the initial join time along a receiver's
 * earliest path.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class JoinVisitor extends AbstractVisitor implements UpVisitor {

    /** Earliest join time */
    protected long earliest = Long.MAX_VALUE;


    /**
     * Default constructor.
     */
    public JoinVisitor() {}


    /**
     * Reset the visitor for a new pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     */
    public void reset( Tree tree ) {

	super.reset( tree );

	// reset the earliest to infinity
	earliest = Long.MAX_VALUE;
    }


    /**
     * Check whether the next receiver should start a visiting path.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>Receiver</code> next receiver
     * @return <code>boolean</code> whether to visit next receiver
     */
    public boolean nextReceiver( Tree tree, Receiver recv ) {

	reset( tree );
	// set earliest to be the first time this receiver was heard from
	earliest = recv.getIntervals().getEarliest();
	
	return( true );
    }

    /**
     * Get a list of parents for the current node applying correct activity
     * semantics.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>TreeNode[]</code> array of parent nodes of currently visited node
     */
    public TreeNode[] getParents( Tree tree, TreeNode node ) {

	// get a list of available parents
	TreeNode parents[] = super.getParents( tree, node );
	TreeNode parent = null;
	Link plink = null;
	long early = Long.MAX_VALUE;
	
	// look for parent with earliest period entry
	for( int p = 0; p < parents.length; p++ ) {
	    // get the link from parent to node
	    Link link = parents[p].getChildLink( node );
	    long first = link.activity.getEarliest();
	    if( first < early ) {
		// save the early time, which parent and which link
		early = first;
		parent = parents[p];
		plink = link;
	    }
	}
	
	// return the parent, setting it's earliest period even earlier
	if( parent != null ) {
	    plink.activity.setEarliest( earliest );
	    if( verbose() )
		plink.activity.print();
	    return( new TreeNode[] { parent } );
	}
	else
	    return( new TreeNode[0] );
    }
}
