/* 
     File : LinkCountVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A visitor used to count the number of links for each receiver 
            along the active tree.
             
*/

package mwalk.visitor;


import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;
import mwalk.core.Config;
import mwalk.util.VisitException;


/**
 * A visitor used to count the number of links for each receiver
 * along the active tree.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class LinkCountVisitor extends PrintVisitor implements DownVisitor {

    /** Default file name if none supplied */
    private static final String DEF_FILENAME = "tree-depth";


    /**
     * Default constructor.
     */
    public LinkCountVisitor() {

	super( DEF_FILENAME );
    }

    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     */
    public LinkCountVisitor( String filename ) {

	super( filename );
    }


    /**
     * Whether to visit prior to visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit prior
     */
    public boolean prefix() {

	return( true );
    }


    /**
     * Print visitor's header info at start of file.
     *
     * @param <code>Tree</code> current tree instance
     */
    protected void printHeader( Tree tree ) {

	super.printHeader( tree );

	ps.println( "#" );
	ps.println( "# receiver\tlinks" );
    }


    /**
     * Initialize visitor.
     *
     * @param <code>Tree</code> current tree instance
     * @return <code>boolean</code> whether to walk again
     */
    public boolean init( Tree tree ) {

	// set source at depth zero
	tree.getSource().data = new Integer( 0 );

	return( super.init( tree ) );
    }


    /**
     * Visit the current node prior to visiting children.
     * This method is called if prefix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node ) throws VisitException {

	// check if node has been seen
	if( ! haveSeen( node ) ) {	    
	    int depth;
	    try {
		// figure out our current depth
		depth = ((Integer)node.data).intValue();
		
	    } catch( Exception e ) {
		throw new VisitException( "node did not have current depth", e );
	    }
	    
	    // print out depth for receiver, else set children's depth
	    if( node instanceof Receiver ) {
		String line = node.getIP() + "\t" + depth;
		
		Config.verbose( this, line );
		printData( tree, line );

	    } else {
		// get a list of active children and set their depth one greater than ours
		TreeNode[] children = node.getActiveChildren( eval );
		depth++;
		
		for( int c = 0; c < children.length; c++ )
		    children[c].data = new Integer( depth );
	    }
	}

	return( null );
    }
}
