/* 
     File : MergeVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A visitor used by the builder to merge the period tables.
             
*/

package mwalk.visitor;


import java.util.Vector;

import mwalk.core.*;
import mwalk.util.VisitException;


/**
 * A visitor used by the builder to merge the period tables.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class MergeVisitor extends AbstractVisitor implements DownVisitor {

    /**
     * Default constructor.
     */
    public MergeVisitor() {}


    /**
     * Whether to visit after visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit after
     */   
    public boolean postfix() {

	return( true );
    }

    /**
     * Reset the visitor for a new pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     */
    public void reset( Tree tree ) {

	super.reset( tree );
	// clear all temporary data in tree
	tree.clearData();
    }


    /**
     * Visit the current node prior to visiting children.
     * This method is called if prefix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node ) {

	// mark node seen and vistited
	markVisited( node );
	// clear activity totals since we're a receiver
	node.data = null;

	return( node.data );
    }

    /**
     * Visit the current node after visiting children.
     * This method is called if postfix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @param <code>Vector</code> list of return values passed back along the path
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node, Vector list ) throws VisitException {

	// only visit once
	if( ! markVisited( node ) ) {
	    // get a list of all downstream links
	    Link links[] = node.getChildLinks();
	    PeriodTable tables[];
	    
	    if( links != null ) {
		tables = new PeriodTable[ links.length ];
		// check activity on each link and merge the link's table
		for( int l = 0; l < links.length; l++ ) {
		    if( verbose() ) {
			Config.verbose( "\tLink from " + node.getIP() + " to " + links[l].node.getIP() );
			links[l].activity.print();
		    }
		    
		    // bound the links activity by the merged activities at the end-point
		    if( links[l].node.data != null )
			links[l].activity.bound( (PeriodTable)links[l].node.data );
		    // merge this links table to save space
		    links[l].activity.merge();
		    // save a reference to the table to report total activity to node's parents
		    tables[l] = links[l].activity;
		    
		    if( verbose() )
			links[l].activity.print();
		}
	    } else
		tables = new PeriodTable[0];
	    
	    // merge all link activity to report total activity to node's parents
	    node.data = PeriodTable.merge( tables );
	} 	    
	
	return( node.data );
    }
}
