/* 
     File : MetricVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : Abstract base class defining the looping behavior of
            most metric visitors.
             
*/

package mwalk.visitor;


import java.util.Vector;

import mwalk.core.*;
import mwalk.util.MetricCount;
import mwalk.util.VisitException;


/**
 * Abstract base class defining the looping behavior of most metric visitors.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public abstract class MetricVisitor extends PrintVisitor implements DownVisitor {

    /** Base of keys in configuration file */
    private static final String CFG_BASE = "MetricVisitor.";
    /** Default file name if none supplied */
    private static final String DEF_FILENAME = "tree-metric";

    protected long time = 0;
    protected double timeval = 0;
    protected double offset = 0;
    protected long periods = 0;

    /**
     * Default constructor.
     */
    public MetricVisitor() {

	super( DEF_FILENAME );
    }

    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     */
    public MetricVisitor( String filename ) {

	super( filename );
    }

    /**
     * Constructor.
     *
     * @param <code>long</code> number of periods to cover session
     */
    public MetricVisitor( long periods ) {

	super( DEF_FILENAME );

	this.periods = periods;
    }

    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     * @param <code>long</code> number of periods to cover session
     */
    public MetricVisitor( String filename, long periods ) {

	super( filename );

	this.periods = periods;
    }


    public long getTime() {

	return( time );
    }

    public long incTime() {

	timeval += offset;
	return( time = (long)timeval );
    }

    public void setTime( double tv ) {

	timeval = tv;
	time = (long)tv;
    }

    public void startTime( Tree tree ) {

	setTime( (double)tree.getStart() );
    }


    /**
     * Whether to visit after visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit after
     */   
    public boolean postfix() {

	return( true );
    }

    /**
     * Initialize the visitor state.
     *
     * @param <code>Tree</code> current tree instance
     */
    public boolean init( Tree tree ) {

	// get period from config file if not already set
	if( periods == 0 ) {
	    periods = cfgInt( CFG_BASE + "periods" );
	    if( periods <= 0 )
		periods = 1;
	}

	// get the start time for the simulation and compute the incremental offset to use
	startTime( tree );
	if( periods > 1 )
	    // set offset to include enough periods hitting both start and stop times
	    offset = (double)(tree.getStop() - tree.getStart()) / (double)(periods - 1);
	else
	    // set offset to push beyond stop time
	    offset = (double)(tree.getStop() - tree.getStart() + 1);

	Config.verbose( this, "\nComputing metrics at time " + time );

	return( super.init( tree ) );
    }

    /**
     * Check whether the visitor wants another pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     * @return <code>boolean</code> whether to walk again
     */
    public boolean again( Tree tree ) {

	try {
	    process( tree );

	} catch( VisitException ve ) {
	    Config.verbose( ve.getMessage() );
	    return( false );
	}

	// reset for the next pass (if more time)
	reset( tree );
	if( incTime() <= tree.getStop() ) {
	    Config.verbose( this, "\nComputing metrics at time " + time );
	    return( true );
	} else {
	    startTime( tree );
	    return( false );
	}
    }


    /**
     * Visit the current node prior to visiting children.
     * This method is called if prefix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node )  throws VisitException {

	// mark node as seen and visited
	markVisited( node );
	// just return a simple metric count for a single receiver (active or not)
	return( new MetricCount( eval.isActive( (Receiver)node, time ) ) );
    }

    /**
     * Visit the current node after visiting children.
     * This method is called if postfix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @param <code>Vector</code> list of return values passed back along the path
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node, Vector list ) throws VisitException {

	MetricCount count = null;

	// check if this node has been visited yet
	if( ! markVisited( node ) ) {

	    // handle the visit (make it easier to sub-class)
	    count = doVisit( tree, node, list );

	    // save count in source
	    if( node instanceof Source )
		node.data = count;
	}

	// return total count for this sub-tree
	return( count );
    }

    /**
     * Get a list of children for the current node applying correct activity
     * semantics.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>TreeNode[]</code> list of active children nodes
     */
    public TreeNode[] getChildren( Tree tree, TreeNode node ) {

	// check if this node has already been seen
	if( ! markSeen( node ) ) {
	    // get all children on active links for the current time
	    return( node.getActiveChildren( eval, time ) );
	} else
	    return( new TreeNode[0] );
    }


    protected abstract MetricCount doVisit( Tree tree, TreeNode node, Vector list ) throws VisitException;

    protected abstract void process( Tree tree ) throws VisitException;
}
