/* 
     File : PrintVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : An abstract visitor used as a bas class for visitors 
            wanting to print results out to a log file.
             
  $Id: PrintVisitor.java,v 1.4 2001/08/22 17:45:07 robertc Exp $
*/

package mwalk.visitor;


import java.util.Date;
import java.io.PrintStream;
import java.io.FileOutputStream;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Config;


/**
 * An abstract visitor used as a bas class for visitors 
 * wanting to print results out to a log file.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public abstract class PrintVisitor extends AbstractVisitor {

    /** Base of keys in configuration file */
    private static final String CFG_BASE = "PrintVisitor.";

    /** Print stream to use for output */
    protected PrintStream ps;
    /** Filename to print to */
    protected String filename = null;
    /** Default file name if none supplied */
    protected String def_filename = "tree-visitor";


    /**
     * Default constructor.
     */
    public PrintVisitor() {}

    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     */
    public PrintVisitor( String filename ) {

	this.def_filename = filename;
    }

    /**
     * Retrieve the current filename.
     *
     * @return <code>String</code> current filename
     */    
    public String getFilename() {

	return( filename );
    }

    /**
     * Set the current filename prior to init().
     *
     * @param <code>String</code> new filename
     */
    public void setFilename( String filename ) {

	this.filename = filename;
    }


    /**
     * Print visitor's header info at start of file.
     *
     * @param <code>Tree</code> current tree instance
     */
    protected void printHeader( Tree tree ) {

	// get the name of the actual class
	String visitor = getClass().getName();
	// print a standard header
	ps.println( "# " + visitor + " output data - " + (new Date()).toString() );
    }

    /**
     * Print a single line of data.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>String</code> line to print
     */
    public void printData( Tree tree, String line ) {

	// print a line of data
	ps.println( line );
    }


    /**
     * Initialize visitor.
     *
     * @param <code>Tree</code> current tree instance
     * @return <code>boolean</code> whether to walk again
     */
    public boolean init( Tree tree ) {

	super.init( tree );

	// get filename from config
	filename = cfgString( CFG_BASE + "filename" );
	if( filename == null )
	    filename = def_filename;

	String visitor = getClass().getName();
	try {
	    ps = new PrintStream( new FileOutputStream( getFilename() ) );
	    printHeader( tree );

	} catch( Exception e ) {
	    Config.verbose( visitor + ".init(): error preparing output file" );
	    return( false );
	}

	return( true );
    }


    /**
     * Get a list of children for the current node applying correct activity
     * semantics.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     */
    public TreeNode[] getChildren( Tree tree, TreeNode node ) {

	// check if this node has already been seen
	if( ! markSeen( node ) ) {
	    // get a list of all child links that are active
	    return( node.getActiveChildren( eval ) );
	} else
	    return( new TreeNode[0] );
    }
}
