/* 
     File : RandomPathVisitor.java
   Author : Robert Chalmers

 Original : November 17, 2000
  Revised : 

  Content : A visitor used to mark the primary path for each randomly selected
            receiver as active.

  $Id: RandomPathVisitor.java,v 1.2 2001/08/22 17:45:07 robertc Exp $
*/

package mwalk.visitor;


import java.util.Enumeration;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;
import mwalk.core.Link;


/**
 * A visitor used to mark the primary path for each randomly selected receiver
 * as active.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class RandomPathVisitor extends AbstractVisitor implements UpVisitor {

    /**
     * Default constructor.
     */
    public RandomPathVisitor() {}


    /**
     * Reset the visitor for a new pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     */
    public void reset( Tree tree ) {
	
	super.reset( tree );

	// clear link activity marks
	for( Enumeration nodes = tree.getNodes(); nodes.hasMoreElements(); )
	    try {
		// get list of links for node
		Link links[] = ((TreeNode)nodes.nextElement()).getChildLinks();
		// clear activiy for each link
		for( int l = 0; l < links.length; l++ )
		    links[l].active = false;

	    } catch( Exception e ) {}
    }


    /**
     * Check whether the next receiver should start a visiting path.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>Receiver</code> next receiver
     * @return <code>boolean</code> whether to visit next receiver
     */
    public boolean nextReceiver( Tree tree, Receiver recv ) {

	// visit if active
	return( recv.isActive() );
    }


    /**
     * Retrieve the list of parents of this node.
     *
     * @param <code>Tree</code> vistited tree 
     * @param <code>TreeNode</code> currently visited node
     * @return <code>TreeNode[]</code> array of parent nodes of currently visited node
     */
    public TreeNode[] getParents( Tree tree, TreeNode node ) {

	// get a list of available parents
	TreeNode parents[] = super.getParents( tree, node );
	
	// look for parent along primary path
	for( int p = 0; p < parents.length; p++ ) {
	    // get the link from parent to node and check if it's primary
	    Link link = parents[p].getChildLink( node );
	    if( link.isPrimary() ) {
		// if already active, then we don't need to continue
		if( ! link.active ) {
		    // save as active link
		    link.active = true;
		    return( new TreeNode[] { parents[p] } );
		} else
		    break;
	    }
	}

	// return no parent nodes
	return( new TreeNode[0] );
    }
}
