/* 
     File : RandomVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : November 17, 2000
            1. Added support for reading defaults from config file.

  Content : A compund visitor which runs other visitors using a random 
            of sampling active receivers.
             
*/

package mwalk.visitor;


import java.util.Vector;
import java.util.Enumeration;


import mwalk.core.Tree;
import mwalk.eval.RandomEvaluator;
import mwalk.util.ReceiverRandomizer;


/**
 * A compund visitor which runs other visitors using a random sampling 
 * of active receivers.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class RandomVisitor extends GlobalVisitor {

    /** Base of keys in configuration file */
    private static final String CFG_BASE = "RandomVisitor.";

    /** Randomizer used select a random set of receivers */
    protected ReceiverRandomizer rand = new ReceiverRandomizer();
    /** Number of receivers to select */
    protected int receivers = 0;
    /** Whether to increment the number of receivers on subsequent rounds */
    protected boolean increment = false;
    /** Amount to increment receivers count by */
    protected int incCount = 0;


    /**
     * Default constructor.
     */
    public RandomVisitor() {

	super( new RandomPathVisitor(), null );
    }


    public RandomVisitor( int rounds, boolean increment, int incCount ) {

	super( new RandomPathVisitor(), null, rounds );

	this.increment = increment;
	this.incCount = incCount;
    }

    public RandomVisitor( DownVisitor dv ) {

	super( new RandomPathVisitor(), dv );
    }

    public RandomVisitor( DownVisitor dv, int rounds, boolean increment, int incCount ) {

	super( new RandomPathVisitor(), dv, rounds );

	this.increment = increment;
	this.incCount = incCount;    
    }


    /**
     * Initialize visitor.
     *
     * @param <code>Tree</code> current tree instance
     * @return <code>boolean</code> whether initialization succeeded
     */
    public boolean init( Tree tree ) {
	
	// try to supply default values from config file
	if( receivers == 0 ) {
	    receivers = cfgInt( CFG_BASE + "receivers" );
	    if( receivers <= 0 )
		receivers = 1;
	}
	if( ! increment )
	    increment = cfgBoolean( CFG_BASE + "increment" );
	if( incCount == 0 ) {
	    incCount = cfgInt( CFG_BASE + "incrementBy" );
	    if( incCount <= 0 )
		incCount = 1;
	}
	
	// setup initial number of receivers
	rand.setActive( receivers );

	// do main initialization
	boolean init = super.init( tree );
	if( init )
	    // set the down visitor's evaluator to check for random activation
	    dv.evaluator( new RandomEvaluator() );

	return( init );
    }


    /**
     * Reset the visitor for a new pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     */
    public void reset( Tree tree ) {
	
	super.reset( tree );

	// randomly select receivers
	rand.randomize( tree );
    }


    /**
     * Determine whether to walk the tree again.
     *
     * @param <code>Tree</code> current tree instance
     * @return <code>boolean</code> whether to walk again
     */
    public boolean again( Tree tree ) {

	// save current state of direction
	boolean down = ! goingUp;
	
	// when down visitor stops, inc active receivers and go again
	if( ! super.again( tree ) )
	    if( increment && rand.incActive() <= rand.maxActive() )
		// reset for another set of rounds with more receivers
		reset( tree );
	    else
		return( false );

	else if( down )
	    // re-randomize receivers for next pass
	    rand.randomize( tree );

	return( true );
    }
}
