/* 
     File : TraceVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A visitor used by the builder to add mtrace paths into the tree.
             
  $Id: TraceVisitor.java,v 1.4 2001/08/22 17:45:07 robertc Exp $
*/

package mwalk.visitor;


import mwalk.core.*;
import mwalk.util.BuildException;
import mwalk.util.VisitException;


/**
 * A visitor used by the builder to add mtrace paths into the tree.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class TraceVisitor extends AbstractVisitor implements UpVisitor {

    /** Associated mtrace object */
    protected MTrace trace;
    /** Current hop count */
    protected int hop = 0;

    /** Reference to parent node */
    TreeNode parent = null;


    public TraceVisitor( MTrace trace ) throws BuildException {

	if( trace.parsed )
	    this.trace = trace;
	else
	    throw new BuildException( "visitor was passed unparsed mtrace" );
    }


    /**
     * Whether to visit prior to visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit prior
     */
    public boolean prefix() {

	return( true );
    }

    /**
     * Reset the visitor for a new pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     */
    public void reset( Tree tree ) {

	super.reset( tree );
	// clear parent reference and hop count
	parent = null;
	hop = 0;
    }


    /**
     * Visit the current node prior to visiting parent.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitUp( Tree tree, TreeNode node ) throws VisitException {

	TreeNode upHop = null;

	// walking will halt if parent is null
	boolean halt = true;
	parent = null;
 
	if( node instanceof Receiver ) {
	    // stop if receiver is not the first hop
	    if( hop > 0 )
		Config.verbose( "TraceVisitor: hit receiver after first hop" );
	    else 
		halt = false;

	} else if( node instanceof Source ) {
	    // stop at source
	    if( hop != trace.hops.length - 1 )
		Config.verbose( "TraceVisitor: hit source before last hop" );

	} else if( hop == 0 ) {
	    // stop if we're starting without a receiver
	    Config.verbose( "TraceVisitor: not starting at receiver" );

	} else if( hop == trace.hops.length - 1 ) {
	    // stop if we're at the end of our hops
	    if( trace.complete )
		Config.verbose( "TraceVisitor: end of complete trace not at source" );

	} else if( trace.hops[hop+1].ip.length() == 0 ) {
	    // we've got an empty hop, just skip it?
	    Config.verbose( "TraceVisitor: skipping empty hop" );
	    hop++;
	    visitUp( tree, node );

	} else if( trace.hops[hop+1].ip.equals( node.getIP() ) ) {
	    // we got a repeat hop, skip to it
	    Config.verbose( "TraceVisitor: skipping duplicate hop" );
	    hop++;
	    visitUp( tree, node );

	} else
	    halt = false;

	if( ! halt ) {
	    // add a new node for the next hop to the tree (if not already there)
	    upHop = tree.addNode( new Router( trace.hops[hop+1].ip ) );
	    // check to make sure that we're not trying to hop to another receiver
	    if( upHop instanceof Receiver ) {
		Config.verbose( "TraceVisitor: hit receiver after first hop" );
		halt = true;
	    }
	}
	
	if( ! halt ) {
	    try {
		// add next hop as this nodes parent router (if not already there)
		parent = node.addParent( upHop );
		TreeNode parents[] = node.getParents();
		long next = Long.MAX_VALUE;
		
		// loop through all other parents and deactivate their link for this time
		for( int p = 0; p < parents.length; p++ )
		    if( ! parents[p].equals( parent ) )
			next = Math.min( next, parents[p].deactivateLink( node, trace.time ) );
		
		// activate the next hop link for this time until the next known trigger time
		parent.activateLink( node, trace.time, next );
		
		// increment the hop count
		hop++;
		
	    } catch( BuildException be ) {
		throw new VisitException( "TraceVisitor error visiting intermediate node", be );
	    } 
	}
	
	return( null );
    }

    /**
     * Get a list of parents for the current node applying correct activity
     * semantics.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>TreeNode[]</code> array of parent nodes of currently visited node
     */
    public TreeNode[] getParents( Tree tree, TreeNode node ) {

	// return the next hop node or empty to stop walking
	if( parent != null )
	    return( new TreeNode[] { parent } );
	else
	    return( new TreeNode[0] );
    }
}
