#! /usr/bin/env perl
#
#     File : generate.pl
#   Author : Robert Chalmers
#
# Original : December 15, 1999
#  Revised :
#
#  Content : main script for producing random tree results
#

package main;

# import external classes/modules
use FileHandle;

# import internal classes/modules
use Mwalk::Args;
use Mwalk::Receiver;
use Mwalk::RTCP;
use Mwalk::MTrace;
use Mwalk::TraceSet;


# enforce strictness
use strict qw( vars refs subs );

# constant pattern for ip address
my $IP = '(?:\d{1,3}\.){3}\d{1,3}';

# command-line arguments used to control execution
my $args = new Mwalk::Args( 1 );
# stats kept for each type of data (rtcp, mtrace, interval)
my %stats = ( counts => { min => undef, max => undef, with => undef, tot => undef, skip => undef },
              skips  => {form => 0, unreachable => 0, traceless => 0}
            );
# list of processed receivers hashed by ip address
my %receivers;


# procedure declarations
use subs qw(initStats printUsage parseArgs initLogs createReceivers saveStats saveData goInteractive);


# prepare the stats array
initStats;
# parse command-line options
parseArgs;
# prepare the output logs
initLogs;

# mark the start time for processing
$args->setStart( time );

# create rtcp and mtrace information for listed receivers
createReceivers;

# mark the stop time for processing
$args->setStop( time );

# output if indicated
saveData;
saveStats if $args->{statFile};

# provide an interactive command line if requested
goInteractive if $args->{interactive};

exit;



#
# initilialize the statistics accumulators
#
sub initStats {

  # initialize stats for interval, rtcp, and mtrace data
  foreach my $stat (keys %{$stats{counts}}) {
    $stats{counts}{$stat} = { interval => 0, rtcp => 0, mtrace => 0 };
  }
  # set minimums arbitrarily high
  $stats{counts}{min}{interval} = $stats{counts}{min}{rtcp} = $stats{counts}{min}{mtrace} = 100000000;
}


#
# print script usage
#
# params
#   optional exit code
#
sub printUsage {

  my $code = shift;
  # default exit code to -1
  $code = -1 unless defined( $code );

  print <<EOU;
mwalk generate [-i|--interactive] [-v|--verbose] [-l|--log]
               [-so|--source=<source>] [-r|--receiver=<rfile>]
               [-c|--config=<cfg>] [-s|--stats=<sfile>] [-d|--dir=<outdir>]
               [-t|--timestamp=<timestamp>] [-bs|--block_size=<bsize>]
               [-f|--fussy] [-a|--append] [-b|--build] [-h|--help]
Options:
  -i  --interactive  run in an interactive mode
  -v  --verbose      print verbose messages
  -l  --log          generate processing logs
  -so --source       host to use as source of mtrace [localhost]
  -r  --receiver     list of receivers to trace [receivers]
  -c  --config       configuration file []
  -s  --stats        generate statistics log [stats]
  -d  --dir          output directory [.]
  -t  --timestamp    beginning timestamp for session [now]
  -bs --block_size   receivers to process before advancing timestamp [10]
  -f  --fussy        use strict parsing
  -a  --append       append to existing log files
  -b  --build        accumulate traces in memory
  -h  --help         print this message
EOU

  # exit with provided code
  exit $code;
}

   
#
# parse the command line options
#
sub parseArgs {

  # preset mtrace output flag and timeout specifically for generation
  $args->{mtraceOutput} = 1;
  $args->{timeout} = 100000;

  # process the command line options
  $args->parse( "interactive!", "verbose!", "output_gen|log!",
		"append!", "build|b!", "fussy!", "help", 
		"source=s", "receiver=s", "stats|s=s", "dir=s",
                "config=s", "timestamp=i", "block_size|bs=i" );

  # sanity check
  $args->{interactive} or -r $args->{recvFile}
                       or print "\nMust specify an input file (-r)\n" and printUsage;
}


#
# prepare the output logs if logging was indicated
#
sub initLogs {

  # return if logging was not requested
  return unless $args->{genOutput};

  my $mode = ($args->{append}) ? ">>" : ">";

  # open log files or quit
  $args->{rtcpHandle} = new FileHandle( "$mode $args->{rtcpFile}" )
                        or die "Couldn't open RTCP log $args->{rtcpFile}: $!\n";
  $args->{mtraceHandle} = new FileHandle( "$mode $args->{mtraceFile}" ) 
                        or die "Couldn't open MTrace log $args->{mtraceFile}: $!\n";

  # print log headers
  print { $args->{rtcpHandle} } "# RTCP PACKET LOG - generated by mwalk\n";
  print { $args->{rtcpHandle} } "# LOG START TIME: ", scalar localtime, "\n";
  print { $args->{rtcpHandle} } "# LOG FORMAT: 1-timestamp 2-ip 3-ssrc 4-reportcount n-fractionlost#n n-jitter#n\n";

  print { $args->{mtraceHandle} } "# ROUTE LOG - generated by mwalk\n";
  print { $args->{mtraceHandle} } "# Started at ", scalar localtime, "\n";

  # clear any temporary logs
  my $mset = new Mwalk::TraceSet( $args );
  $mset->clearLogs();
}


#
# add a new receiver if that receiver is reachable
# 
# params
#   ip address of receiver
#
# return
#   whether the receiver was added
# 
sub createReceiver {

  my $ip = shift;

  # ensure receiver not already in list
  unless( exists $receivers{$ip} ) {
    # check whether the receiver is reachable
    if( system( "$args->{ping} $args->{pingFlags} $ip > /dev/null" ) == 0 ) {
      print "Adding receiver for $ip\n" if $args->{verbose};
      # create a new receiver
      $receivers{$ip} = new Mwalk::Receiver( $ip, $args );
      return 1;
    } else {
      print "Receiver $ip unreachable\n" if $args->{verbose};
      $stats{skips}{unreachable}++;      
    }
  }

  return 0;
}


#
# create a faked rtcp packet for this receiver
#
# params
#   ip address of receiver
#
sub createRTCP {

  my $ip = shift;
  my $rtcp = new Mwalk::RTCP;
  
  print "Creating RTCP entry for receiver $ip at time $args->{timestamp}\n" if $args->{verbose};

  # fill in the receiver and timestamp info
  $rtcp->generate( $ip, $args );
  # add the rtcp packet to the receiver
  $receivers{$ip}->add( $rtcp );
}


#
# generate an mtrace for this receiver
#
# params
#   ip address of receiver
#
sub createMTrace {  

  my $ip = shift;
  my ($mtrace, $mset) = (undef, new Mwalk::TraceSet( $args ));
  
  print "Starting MTrace processing for $ip\n" if $args->{verbose};
    
  # generate and parse an mtrace for the given receiver
  $mtrace = $mset->generate( $ip, "mtrace" );

  # check if parsed correctly, add to receiver or remove receiver entirely
  if( defined( $mtrace ) ) {
    $receivers{$ip}->add( $mtrace );
  } else {
    delete $receivers{$ip};
    $stats{skips}{traceless}++;
    $stats{count}{skip}{mtrace}++;
  } 
}
  

#
# take a list of receivers and try to create mtraces for each
#
sub createReceivers {

  my ($fh, $line, $ip);
  my $reps = 1;

  # open receivers file
  $fh = new FileHandle( "< $args->{recvFile}" ) or die "Couldn't open receivers file: $!\n";

  print "Starting receiver processing for $args->{recvFile}\n" if $args->{verbose};
  print "Enter number of reps:" and $reps = <STDIN> + 0 if $args->{interactive};

  # parse each line as a receiver's ip address
  while( $line = <$fh> and $reps ) {
    # ignore comments and blank lines
    next if $line =~ m/^\s*\#/o or $line =~ m/^\s*$/o;

    # look for ip address in line
    if( ($ip) = ($line =~ m/($IP)/o) ) {
      # try to create a new receiver with rtcp and mtrace data
      createRTCP( $ip ), createMTrace( $ip ) if createReceiver( $ip );

      # if we've added enough receivers to fill the current block advance the timestamp
      $args->advanceTime( scalar keys %receivers );

    } else {
      print "Didn't find ip address on line\n";
      $stats{skips}{form}++;
    }

  } continue {
    print "Enter number of reps:" and $reps = <STDIN> + 0 if $args->{interactive} and ! --$reps;
  }

  # clear any temporary logs
  my $mset = new Mwalk::TraceSet( $args );
  $mset->clearLogs();
}


#
# save statistics about data collection
#
sub saveStats {
 
  my ($first, $last, $recvs) = (0, 0, 0);
  my ($byes, $byed, $patches, $patched, $completes, $completed) = (0, 0, 0, 0, 0, 0);

  print "Compiling statistics\n" if $args->{verbose};

  # compute stats for each receiver
  my ($ip, $recv);
  while( ($ip, $recv) = each %receivers ) {
    # compute stats for rtcp and mtrace data
    foreach my $data ("interval", "rtcp", "mtrace") {
      my $len = $recv->size( $data );
      $stats{counts}{min}{$data} = $len if $len < $stats{counts}{min}{$data};
      $stats{counts}{max}{$data} = $len if $len >  $stats{counts}{max}{$data};
      $stats{counts}{with}{$data}++ if $len;
      $stats{counts}{tot}{$data} += $len;
    }
    # track start and stop time of entire session
    $first = $recv->firstTime() if ! $first or $recv->firstTime() < $first;
    $last = $recv->lastTime() if $recv->lastTime() > $last;
    # keep count of byes processed and traces complete and/or patched
    $byes += $recv->byes();
    $byed++ if $recv->byes();
    $patches += $recv->patches();
    $patched++ if $recv->patches();
    $completes += $recv->completes();
    $completed++ if $recv->completes();
    # count the number of receivers
    $recvs++;
  }

  # open a new file for writing if stat file specified
  my $fh = new FileHandle( (($args->{append}) ? ">>" : ">") . " $args->{statFile}" ) 
           or die "Can't open stat file $args->{statFile}: $!\n";
  
  # print runtime arguments
  $args->save( $fh );
  # print runtime times
  print $fh "start:", scalar localtime $args->{start}, "\n";
  print $fh "stop:", scalar localtime $args->{stop}, "\n";
  print $fh "elapsed:", $args->{stop} - $args->{start}, " sec.\n\n";

  # print stats
  print $fh "STATS:\n";
  print $fh "first:$first\nlast:$last\n";
  print $fh "receivers:$recvs\n";
  print $fh "byes:$byes from $byed\npatches:$patches from $patched\n";
  print $fh "complete:$completes from $completed\n";
  print $fh "SKIPS:\n";
  foreach my $skip (keys %{$stats{skips}}) {
    print $fh "$skip:$stats{skips}{$skip}\n";
  }
  # print individual stats for rtcp and mtrace data
  foreach my $data ("interval", "rtcp", "mtrace") {
    print $fh "\n\U$data:\n";
    print $fh "min:$stats{counts}{min}{$data}\nmax:$stats{counts}{max}{$data}\nwith:$stats{counts}{with}{$data}\n";
    print $fh "avg:", (($recvs) ? $stats{counts}{tot}{$data} / $recvs : 0), "\ntot:$stats{counts}{tot}{$data}\n";
    print $fh "skip:$stats{counts}{skip}{$data}\n";
  }

  # close the stats file
  close $fh;
}


#
# save collected data into output file(s)
#
sub saveData {

  my ($with, $without, $mode);
  
  # open tabulation files to record which receivers have mtraces
  $mode = ($args->{append}) ? ">>" : ">";
  $with = new FileHandle( "$mode $args->{dir}receivers.with" ) or die "Couldn't open with receivers file: $!\n";
  $without = new FileHandle( "$mode $args->{dir}receivers.without" ) or die "Couldn't open without receivers file: $!\n";

  # save data for each receiver
  my ($ip, $recv, $fh);
  while( ($ip, $recv) = each %receivers ) {     
    print "Saving data for receiver $ip\n" if $args->{verbose};
    # save receiver data in individual files named by ip  
    $recv->save();
    # save whether this receiver has mtrace data available
    $fh = ($recv->mtraces()) ? $with : $without;
    print $fh $recv->ip(), "\n";
  }

  close $with;
  close $without;
}



#
# enter an interactive mode to allow on-line querying
#
sub goInteractive {

  my ($line, $cmd) = (undef, "");

  print "Entering interactive mode\n" if $args->{verbose};
  # print prompt
  print "\ngenerate> ";

  # loop around until user exits
  while( $line = <STDIN> ) {
    # append the line to the command
    $cmd .= $line;
    
    # check for line continuation
    print "> " and next if $line =~ m/\\\w*$/;
    # check for exit command
    last if $cmd =~ m/^\w*exit\w*(\\\w*)*$/;
    
    # evaluate command if something passed
    print "error: $@\n" if $cmd !~ m/^\w*$/ and ! defined( eval $cmd );

    # clear command and reprint prompt
    $cmd = "";
    print "generate> ";
  }
}
