/* 
     File : Animator.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A runnable class that provides loading and launching for specific
            visitors listed on the command-line.
             
  $Id: Animator.java,v 1.6 2001/10/19 19:01:43 robertc Exp $
*/

package mwalk.app;


import java.util.Date;

import mwalk.core.*;
import mwalk.visitor.Visitor;
import mwalk.visitor.UpVisitor;
import mwalk.visitor.DownVisitor;
import mwalk.util.VisitException;


/**
 * A runnable class that provides loading and launching for specific
 * visitors listed on the command-line.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class Animator {

    protected Tree tree;


    /**
     * Constructor.
     *
     * @param <code>Tree</code> tree to animate
     */
    public Animator( Tree tree ) {
	
	this.tree = tree;
    }


    /**
     * Animate the tree with the specified visitor.
     *
     * @param <code>String</code> full classname of visitor
     */
    public void animate( String visitClass ) {

	try {	    
	    Visitor visitor = null;

	    try {
		// try to instantiate with passed name
		visitor = (Visitor)Class.forName( visitClass ).newInstance();

	    } catch( ClassNotFoundException cne ) {
		// try with standard visitor base package
		visitor = (Visitor)Class.forName( "mwalk.visitor." + visitClass ).newInstance();
	    }

	    Date start = new Date();
	
	    Config.verbose( "\nAnimating " + visitClass );

	    // initialize visitor
	    if( visitor.init( tree ) )
		do {
		    // either walk down or up the tree and visit each node
		    if( visitor instanceof DownVisitor && ((DownVisitor)visitor).walkDown() )
			tree.walkDown( (DownVisitor)visitor );
		    else if( visitor instanceof UpVisitor && ((UpVisitor)visitor).walkUp() )
			tree.walkUp( (UpVisitor)visitor );
		// check if visitor wants to go again
		} while( visitor.again( tree ) );

	    Config.verbose( "\nTree walked in " + (((new Date()).getTime() - start.getTime()) / 1000.0) + " sec." );

	} catch( VisitException ve ) {
	    Config.verbose( "Exception caught animating " + visitClass + "\n\t" + ve.getMessage() );
	} catch( Exception e ) {
	    Config.verbose( "Couldn't animate " + visitClass );
	    Config.verbose( e );
	}
    }


    /**
     * Main application method.
     *
     * @param <code>String[]</code> command line arguments.
     */
    public static void main( String args[] ) {

	try {
	    // parse the command-line arguments
	    Config.parseArgs( args );
	    // build the tree and create a new animator
	    Animator anim = new Animator( (new Builder()).build() );
	    
	    // walk through the tree for each listed visitor
	    for( int i = 0; i < Config.visitors.length; i++ )
		anim.animate( Config.visitors[i] );

	} catch( Exception e ) {
	    e.printStackTrace();
	    System.exit( -1 );
	}
    }
}
