/* 
     File : MTrace.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A class representing a single mtrace.
            
  $Id: MTrace.java,v 1.6 2001/10/19 18:58:25 robertc Exp $ 
*/

package mwalk.core;


import java.io.BufferedReader;

import mwalk.util.BuildException;
import mwalk.util.ConfigException;


/**
 * A class representing a single mtrace.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class MTrace {

    /** Whether the mtrace has been parsed */
    public boolean parsed = false;
    /** Whether the trace was complete */
    public boolean complete = false;
    /** Whether the trace type is gateway */
    public boolean gateway = false;
    /** Timestamp of the trace */
    public long time = 0;
    /** List of hops in the trace */
    public Hop hops[] = null;


    /** 
     * Default constructor.
     */
    public MTrace() {}


    /**
     * Constructor.
     *
     * @param <code>BufferedReader</code> reader to use for parsing
     * @exception <code>BuildException</code> if parsing failed
     * @exception <code>ConfigException</code> if the trace has the wrong source
     */
    public MTrace( BufferedReader ir ) throws BuildException, ConfigException {
	// parse the next mtrace
	parse( ir );
    }



    /**
     * Parse the next trace from the supplied reader.
     *
     * @param <code>BufferedReader</code> reader to use for parsing
     * @exception <code>BuildException</code> if parsing failed
     * @exception <code>ConfigException</code> if the trace has the wrong source
     */
    public void parse( BufferedReader ir ) throws BuildException, ConfigException {

	String line, label, value;
	int marker;

	try {
	    while( (line = ir.readLine()) != null && ! line.startsWith( "TRACE" ) ) {}
	    if( line == null )
		return;

	    // we should be at the start of the trace
	    complete = line.substring( line.indexOf( ':' ) + 1 ).equals( "complete" );

	    while( (line = ir.readLine()) != null ) {
		// parse the label/value pairs
		marker = line.indexOf( ':' );
		label = line.substring( 0, marker ).trim();
		value = line.substring( marker+1 ).trim();
		
		if( label.equals( "source" ) ) {
		    // make sure this is for the correct source
		    // just leave if source is empty (ignore the mtrace)
		    if( value.length() == 0 )
			return;
		    else if( ! Config.source.equals( value ) )
			throw new ConfigException( "trace has wrong source" );

		} else if( label.equals( "type" ) ) {
		    // save whether mtrace is a gateway trace
		    gateway = value.equals( "gateway" );
		} else if( label.equals( "start" ) ) {
		    // save the start time of the mtrace
		    time = Long.parseLong( value );
		} else if( label.equals( "hops" ) ) {
		    // parse the individual hops
		    parseHops( ir, Integer.parseInt( value ) );
		    break;
		}
	    }

	} catch( ConfigException ce ) {
	    throw ce;
	} catch( BuildException be ) {
	    throw be;
	} catch( Exception e ) {
	    throw new BuildException( "error parsing mtrace", e );
	}

	parsed = true;
    }    


    /**
     * Parse the list of hops for this trace.
     *
     * @param <code>BufferedReader</code> reader to use for parsing
     * @param <code>int</code> number of hop currently parsing
     * @exception <code>BuildException</code> if parsing failed
     */
    public void parseHops( BufferedReader ir, int hopCount ) throws BuildException {

	String line;

	hops = new Hop[ hopCount ];

	try {
	    for( int h = 0; h < hopCount; h++ ) {
		if( (line = ir.readLine()) == null )
		    throw new BuildException( "error parsing hops - not enough" );

		hops[h] = new Hop( line );
	    }

	} catch( BuildException be ) {
	    throw be;
	} catch( Exception e ) {
	    throw new BuildException( "error parsing hops", e );
	}
    }
}
