/* 
     File : PeriodTable.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A class representing the cummulative activity for a link.
  
  $Id: PeriodTable.java,v 1.6 2001/10/19 18:58:25 robertc Exp $           
*/

package mwalk.core;


import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;
import java.util.Hashtable;
import java.io.Serializable;
import java.io.BufferedReader;
import java.io.ObjectInputStream;
import java.io.IOException;

import mwalk.util.BuildException;


/**
 * A class representing the cummulative activity for a link.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class PeriodTable implements Serializable {

    /** Serialize version ID */
    static final long serialVersionUID = 9136775705866627006L;

    /** List of period entries */
    protected Vector entries = new Vector( 3, 5 );
    /** Temporary data that can be stored in table */
    public transient Hashtable data = new Hashtable();

    
    /**
     * Default constructor.
     */
    public PeriodTable() {}


    /**
     * Retrieve the earliest time in the period table.
     *
     * @return <code>long</code> earliest time in the table or Long.MAX_VALUE if no earliest time available
     */
    public long getEarliest() {

	try {
	    return( ((PeriodEntry)entries.firstElement()).start );

	} catch( NoSuchElementException nse ) {}

	return( Long.MAX_VALUE );
    }


    /**
     * Set the earliest time in the period table if not already earlier.
     *
     * @param <code>long</code> earliest time to set
     */
    public void setEarliest( long time ) {

	try {
	    PeriodEntry first = (PeriodEntry)entries.firstElement();
	    if( time < first.start )
		first.start = time;

	} catch( NoSuchElementException nse ) {}
    }


    /**
     * Retrieve the latest time in the period table.
     *
     * @return <code>long</code> latest time in the table or Long.MAX_VALUE if no latest time available
     */
    public long getLatest() {

	try {
	    return( ((PeriodEntry)entries.lastElement()).stop );

	} catch( NoSuchElementException nse ) {}

	return( Long.MAX_VALUE );
    }


    /**
     * Retrieve the total active time in the period table.
     *
     * @param <code>bound</code> maximum stop time considered
     * @return <code>long</code> total active time
     */
    public long getDuration( long bound ) {

	long duration = 0;

	for( Enumeration enum = entries.elements(); enum.hasMoreElements(); ) 
	    try {
		PeriodEntry entry = (PeriodEntry)enum.nextElement();
		duration += entry.duration( bound );
		
	    } catch( NoSuchElementException nse ) {}

	return( duration );
    }


    /**
     * Check whether a specific time falls within the table periods.
     *
     * @param <code>long</code> time to check for
     * @return <code>boolean</code> whether the time fell withing the tables periods
     */
    public boolean check( long time ) {

	for( Enumeration enum = entries.elements(); enum.hasMoreElements(); ) 
	    try {
		PeriodEntry entry = (PeriodEntry)enum.nextElement();
		if( entry.contains( time ) )
		    // it's in this entry's range
		    return( true );
		
	    } catch( NoSuchElementException nse ) {}
	
	return( false );
    }

    
    /**
     * Add a new time period into the table, without merging adjacent entries.
     * Start times need to be kept so that further adds will respect them.
     *
     * @param <code>long</code> start time of period
     * @param <code>long</code> stop time of period
     * @return <code>long</code> actual stop time, possibly reduced due to earlier explicit start times 
     */
    public long add( long start, long stop ) {

	PeriodEntry prior = null, entry = null;

	for( int i = 0; i < entries.size(); i++ ) 
	    try {
		prior = entry;
		entry = (PeriodEntry)entries.get( i );
		
		if( start < entry.start ) {
		    // it should go before this entry
		    // possibly bound the stop time by the next known start time
		    stop = Math.min( stop, entry.start );
		    // add the entry
		    entries.add( i, new PeriodEntry( start, stop ) );
		    if( prior != null )
			// possibly bound the prior entry's stop by our start time
			prior.stop =  Math.min( prior.stop, start );
		    // return the actual stop time used
		    return( stop ); 

		} else if( start == entry.start ) 
		    // a repeat of this entry, so bound the stop time
		    return( entry.stop = Math.min( entry.stop, stop ) );

	    } catch( ArrayIndexOutOfBoundsException obe ) {
		Config.verbose( "Broke array bound in PeriodTable.add()" ); 
	    }

	// it's beyond all known periods, so add it
	entries.add( new PeriodEntry( start, stop ) );
	if( entry != null && entry.stop > start )
	    // the last entry of the list may need to be bounded
	    entry.stop = start;

	// return the actual stop time
	return( stop );
    }


    /**
     * Bound the period table with an explicit stop time.
     * This happens when a child alerts a parent that another link was used at this time.
     *
     * @param <code>long</code> explicit stop time
     * @return <code>long</code> next explicit start time beyond this stop
     */
    public long bound( long stop ) {

	PeriodEntry entry = null, next = null;

	for( int i = 0; i < entries.size(); i++ ) 
	    try {
		entry = (i == 0) ? (PeriodEntry)entries.get( i ) : next;
		next = (i + 1 < entries.size()) ? (PeriodEntry)entries.get( i + 1 ) : null;

		if( stop < entry.start )
		    // happened before this entry, so return our start time as the next known
		    return( entry.start );
		else if( next == null )
		    // we've hit the end, so bound the final stop time
		    entry.stop = Math.min( entry.stop, stop );
		else if( stop < next.start ) {
		    // the stop is prior to our next entry's start, so bound this stop
		    entry.stop = Math.min( entry.stop, stop );
		    // return the next entry's start time
		    return( next.start );
		}
	    } catch( ArrayIndexOutOfBoundsException obe ) {
		Config.verbose( "Broke array bound in PeriodTable.bound()" ); 
	    }

	// we don't have a next start time
	return( Long.MAX_VALUE );
    }


    /**
     * Bound all entries by those from another table (child's table).
     * A parent may have an exagerrated view of the activity on a particular link, so
     * this allows us to constrain the table to what the child reports as it's actual needs.
     *
     * @params <code>PeriodTable</code> table to use for bounding
     */ 
    public void bound( PeriodTable bounder ) {

	PeriodEntry entry = null, bound = null;
	int b = 0, i =0;

	while( entry != null || i < entries.size() ) 
	    try {
		entry = (entry == null) ? (PeriodEntry)entries.get( i++ ) : entry;
		bound = (bound == null && b < bounder.entries.size()) ? (PeriodEntry)bounder.entries.get( b++ ) : bound;
		
		// do we have a bound to work with
		if( bound != null ) {
		    if( bound.stop < entry.start ) {
			// bounding entry is prior to this entry, just ignore it
			bound = null;
			continue;
		    }

		    if( bound.start <= entry.start ) {
			if( bound.stop <= entry.stop ) {
			    // bound this entries stop time and advance the bound
			    entry.stop = bound.stop;
			    bound = null;
			}
		    } else if( bound.start < entry.stop ) {
			// bound this enties start time and advance the entry
			entry.start = bound.start;

			if( bound.stop <= entry.stop ) {
			    // bound this entries stop time and advance the bound
			    entry.stop = bound.stop;
			    bound = null;
			}
		    } else
			// otherwise the bound is passed this entry, we should remove it (let merge take care it)
			entry.stop = entry.start;
		} else
		    // otherwise the bound is exhausted, we should remove the entry (let merge take care it)
		    entry.stop = entry.start;

		// advance the entry
		entry = null;
		
	    } catch( ArrayIndexOutOfBoundsException obe ) {
		Config.verbose( "Broke array bound in PeriodTable.bound()" ); 
	    }
    }


    /**
     * Merge the activity table with no specific bound on the final stop time.
     *
     * @return <code>long</code> actual final stop time
     */
    public long merge() {

	return( merge( Long.MAX_VALUE ) );
    }


    /**
     * Merge the activity table with a specific bound on the final stop time.
     *
     * @param <code>long</code> bound on final stop time
     * @return <code>long</code> actual final stop time
     */
    public long merge( long maxStop ) {

	PeriodEntry entry = null, next = null;

	for( int i = 0; i < entries.size(); ) 
	    try {
		entry = (next == null) ? (PeriodEntry)entries.get( i ) : next;
		next = (i + 1 < entries.size()) ? (PeriodEntry)entries.get( i + 1 ) : null;
		
		if( next != null && entry.stop >= next.start ) {
		    // our stop time overlap the next entry, so combine
		    next.start = entry.start;
		    entries.remove( i );
		} else if( entry.start == entry.stop ) {
		    // our start and stop times match (null time period), so remove
		    entries.remove( i ); 
		    // if we removed the last element then set entry back one so that 
		    //   its stop time can be bounded(PeriodEntry)entries.get( i )
		    if( i == entries.size() && i > 0 )
			entry = (PeriodEntry)entries.get( i - 1 );
		    else
			entry = null;
		} else 
		    // move on to the next entry
		    i++;

	    } catch( ArrayIndexOutOfBoundsException obe ) {
		Config.verbose( "Broke array bound in PeriodTable.merge()" ); 
	    }

	if( entry != null ) {
	    // bound the final stop time
	    entry.stop = Math.min( entry.stop, maxStop );
	    return( entry.stop );
	} else
	    return( Long.MAX_VALUE );
    }


    /**
     * Merge periods from another table into this one.
     * This is used to build a single table representing the total activity on all links of a node.
     * This total activity is then used by its parents to bound their tables.
     *
     * @param <code>PeriodTable</code> table to merge into this one
     */
    public void merge( PeriodTable merger ) {

	PeriodEntry prev = null, entry = null, merge = null;
	int m = 0;

	for( int i = 0; i < entries.size() && (merge != null || m < merger.entries.size()); ) 
	    try {
		prev = (entry != null) ? entry : prev;
		entry = (PeriodEntry)entries.get( i );
		merge = (merge == null) ? (PeriodEntry)merger.entries.get( m++ ) : merge;

		if( prev != null && prev.stop > entry.stop ) {
		    // the previous entry stops later than the this one, so just yank it
		    entries.remove( i );
		    entry = null;
		    continue;
		} else if( merge.stop < entry.start ) {
		    // the new period takes place prior to this entry, so insert it 
		    prev = new PeriodEntry( merge );
		    entries.add( i, prev );
		    entry = merge = null;
		} else if( merge.start < entry.stop ) {
		    // now we have overlapping periods
		    if( merge.start < entry.start )
			// the new one is earlier, so use its start time
			entry.start = merge.start;
		    // use the latest stop time
		    entry.stop = Math.max( entry.stop, merge.stop );
		    merge = null;
		}
		i++;

	    } catch( ArrayIndexOutOfBoundsException obe ) {
		Config.verbose( "Broke array bound in PeriodTable.merge()" ); 
	    }

	// fill in any left over periods
	for( m = (merge == null) ? m : m - 1; m < merger.entries.size(); m++ )
	    entries.add( new PeriodEntry( (PeriodEntry)merger.entries.get( m ) ) );

	// compact the table
	merge();
    }


    /**
     * Create a new table that contains the total activity of all component tables.
     *
     * @param <code>PeriodTable[]</code> list of component tables to merge
     * @return <code>PeriodTable</code> new table with total activity
     */
    public static PeriodTable merge( PeriodTable[] tables ) {

	PeriodTable merged = new PeriodTable();

	// merge each table into the new one
	for( int t = 0; t < tables.length; t++ )
	    merged.merge( tables[t] );  

	return( merged );
    }


    /**
     * Load a period table from a file.
     *
     * @param <code>BufferedReader</code> reader to use to parse file 
     */
    public void load( BufferedReader br ) throws BuildException {

	String line;
	long start, stop;
	boolean bye;

	try {
	    while( (line = br.readLine()) != null && line.trim().length() > 0 ) {
		StringTokenizer toker = new StringTokenizer( line );

		// parse start and stop times, and whether the stop was explicit
		start = Long.parseLong( toker.nextToken() );
		stop = Long.parseLong( toker.nextToken() );
		bye = toker.nextToken().equals( "B" );

		// keep merge from wiping this entry out
		if( stop == start )
		    stop++;

		add( start, stop );
	    }
	     
	    merge();

	} catch( Exception e ) {
	    throw new BuildException( "error loading receiver periods", e );
	}
    }


    /**
     * Print the contents of the period table.
     */
    public void print() {

	Config.verbose( "\tActivity:" );

	try {
	    for( Enumeration enum = entries.elements(); enum.hasMoreElements(); ) {
		PeriodEntry entry = (PeriodEntry)enum.nextElement();
		Config.verbose( "\t\t" + entry.start + "\t" + entry.stop );
	    }
	} catch( Exception e ) {
	    Config.verbose( "\tError printing: " + e.getMessage() );
	}
    }


    /**
     * Clear temporary data for table and each entry
     */
    public void clearData() {

	// clear data for each associated entry
	for( Enumeration ents = entries.elements(); ents.hasMoreElements(); )
	    try {
		((PeriodEntry)ents.nextElement()).clearData();

	    } catch( Exception e ) {}

	// clear local data
	data.clear();
    }


    /**
     * Initialize transient variables after reading the object in.
     *
     * @param <code>ObjectInputStream</code> input stream
     * @exception <code>IOException</code> if object not read correctly
     */
    private void readObject( ObjectInputStream in ) throws IOException, ClassNotFoundException {
	
	// handle default read
	in.defaultReadObject();

	// initialize transient variables
	data = new Hashtable();
    }
}
