/* 
     File : DegreeCount.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A class used by the degree count visitor to track the degree data.
             
*/

package mwalk.util;


import mwalk.core.*;


/**
 * A class used by the degree count visitor to track the degree data.
 *
 * @author Robert Chalmers00
 * @version 1.0 
 */
public class DegreeCount {

    /** Whether this count comes from a receiver */
    public boolean fromReceiver = false;
    /** Number of receivers downstream */
    public int receivers = 0;
    /** Number of active links */
    public int links = 0;
    /** Number of active links to receivers */
    public int leafs = 0;
    /** Optional data to be passed back with count */
    public Object data = null;

    /**
     * Default constructor.
     */
    public DegreeCount() {}

    /**
     * Constructor.
     *
     * @param <code>Object</code> optional data to pass along with count
     */
    public DegreeCount( Object data ) {
	
	// save data
	this.data = data;
    }

    /**
     * Constructor.
     *
     * @param <code>boolean</code> whether receiver is active
     */
    public DegreeCount( boolean active ) {
	
	fromReceiver = true;
	// if the receiver is active, mark it
	if( active )
	    receivers++;
    }

    /**
     * Constructor.
     *
     * @param <code>boolean</code> whether receiver is active
     * @param <code>Object</code> optional data to pass along with count
     */
    public DegreeCount( boolean active, Object data ) {
	
	fromReceiver = true;
	// if the receiver is active, mark it
	if( active )
	    receivers++;
	// save data
	this.data = data;
    }

    /**
     * Constructor.
     *
     * @param <code>int</code> number of downstream receivers
     * @param <code>int</code> total number of links
     * @param <code>int</code> number of links connected to receivers
     * @param <code>Object</code> optional data to pass along with count
     */        
    public DegreeCount( int receivers, int links, int leafs, Object data ) {
	
	this.receivers = receivers;
	this.links = links;
	this.leafs = leafs;
	this.data = data;
    } 


    /**
     * Add a degree count to a running total.
     * This simply increments the link counts if some receivers are downstream.
     *
     * @param <code>DegreeCount</code> count from downstream neighbor
     */
    public void add( DegreeCount count ) {
	
	receivers += count.receivers;
	if( count.receivers > 0 ) {
	    links++;
	    if( count.fromReceiver )
		leafs++;
	}
    }

    /**
     * Add just the number of downstream receivers to a running total.
     *
     * @param <code>DegreeCount</code> count from downstream neighbor
     */
    public void addReceivers( DegreeCount count ) {
	
	receivers += count.receivers;
    }

    /**
     * Print a count as a tab-separated string.
     */
    public String toString() {
	
	return( receivers + "\t" + links + "\t" + leafs );
    }
}
