/* 
     File : CollapseVisitor.java
   Author : Robert Chalmers

 Original : October 18, 2001
  Revised : November 17, 2000
            1. Added support for reading config file values in number of 
               formats.

  Content : A visitor that collapses the primary tree based on matching data
	    at each node.  The type of data referenced can be specified in the
	    config file.
             
  $Id: CollapseVisitor.java,v 1.2 2001/12/19 19:33:50 robertc Exp $
*/	

package mwalk.visitor;


import java.util.Vector;
import java.util.Hashtable;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;
import mwalk.core.Link;
import mwalk.core.Config;
import mwalk.util.VisitException;


/**
 * A visitor that collapses the primary tree based on matching data at
 * each node.  The type of data referenced can be specified in the
 * config file.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class CollapseVisitor extends AbstractVisitor implements DownVisitor {

    /** Base of keys in configuration file */
    private static final String CFG_BASE = "CollapseVisitor.";

    /** Type of data - used as node's hash key */
    protected String type = "data";    


    /**
     * Default constructor.
     */
    public CollapseVisitor() {}

    
    /**
     * Whether to visit after visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit after
     */   
    public boolean postfix() {

	return( true );
    }


    /**
     * Initialize the visitor state.
     *
     * @param <code>Tree</code> current tree instance
     * @return <code>boolean</code> whether to walk again
     */
    public boolean init( Tree tree ) {

	super.init( tree );

	// get type from config
	String t = cfgString( CFG_BASE + "type" );
	if( t != null )
	    type = t;

	// setup a primary path evaluator
	evaluator( new mwalk.eval.PrimaryPathEvaluator() );	

	return( true );
    }


    /**
     * Transfer children from one node to another.
     *
     * @param <code>TreeNode</code> node to transfer from
     * @param <code>TreeNode</code> node to transfer to
     */
    protected void transfer( TreeNode src, TreeNode dst ) {
	
	// get a list of active links for the source
	Link[] slinks = src.getActiveLinks( eval );
	
	// transfer each child and mark current link as inactive
	for( int l = 0; l < slinks.length; l++ ) {
	    // create a new link from dst to child of src
	    slinks[l].node.addParent( dst );
	    // mark the new link as primary and downgrade the old one
	    dst.getChildLink( slinks[l].node ).primary = true;
	    slinks[l].primary = false;
	}
    }


    /* DownVisitor Methods */

    /**
     * Visit the current node after visiting children.
     * This method is called if postfix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @param <code>Object</code> list of return values passed back along the path
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node, Vector list ) throws VisitException {

	// get a reference to our own data item to collapse like nodes
	Object data = node.data.get( type );
	if( data == null )
	    throw new VisitException( "error collapsing, node has no data value" );

	// mark as visited - only once
	if( markVisited( node ) )
	    return( null );

	// get a list of active links
	Link[] links = node.getActiveLinks( eval );
	// first subsume all child nodes with same data value
	for( int l = 0; l < links.length; l++ ) {
	    // get child but don't include if it's a receiver
	    TreeNode child = links[l].node;
	    if( child instanceof Receiver )
		continue;

	    Object cdata = child.data.get( type );
	    Config.verbose( this, "Parent " + node.getIP() + " " + type + "=" + data.toString() + ", Child " + type + "=" + cdata.toString() );
	    // check if data matches
	    if( cdata != null && cdata.equals( data ) ) {
		// inherit each grandchild and mark link as inactive
		transfer( child, node );
		links[l].primary = false;
		Config.verbose( this, "Inheriting from " + child.getIP() );
	    }
	}

	// get a new list of active links
	links = node.getActiveLinks( eval );
	Hashtable reps = new Hashtable();
	// combine like children into a single representative node
	for( int l = 0; l < links.length; l++ ) {
	    // get child but don't include if it's a receiver
	    TreeNode child = links[l].node;
	    if( child instanceof Receiver )
		continue;

	    // is there a representative for this set
	    Object cdata = child.data.get( type );
	    Config.verbose( this, "Parent " + node.getIP() + " " + type + "=" + data.toString() + ",Child " + type + "=" + cdata.toString() );
	    TreeNode rep = (TreeNode)reps.get( cdata );
	    if( rep == null ) {
		// make this node the representative
		reps.put( cdata, child );
		Config.verbose( this, "Making " + child.getIP() + " a rep" );
	    } else {
		// transfer grandchildren to representative
		transfer( child, rep );
		links[l].primary = false;		
		Config.verbose( this, "Combining " + child.getIP() + " with " + rep.getIP());

	    }
	}

	return( null );
    }


    /**
     * Retrieve the list of children of this node.
     *
     * @param <code>Tree</code> vistited tree 
     * @param <code>TreeNode</code> currently visited node
     * @return <code>TreeNode[]</code> array of child nodes of currently visited node
     */
    public TreeNode[] getChildren( Tree tree, TreeNode node ) {
	
	// check if this node has already been seen
	if( ! markSeen( node ) )
	    // get a list of all child links that are active
	    return( node.getActiveChildren( eval ) );
	else
	    return( new TreeNode[0] );
    }
}
