/* 
     File : DegreeCountVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A visitor used to count the degree of each node along the
            active tree.
             
*/

package mwalk.visitor;


import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;
import mwalk.core.Config;
import mwalk.util.DegreeCount;
import mwalk.util.VisitException;


/**
 * A visitor used to count the degree of each node along the
 * active tree. This visitor assumes that the node's data contain
 * the current depth, otherwise the depth will be reported as zero.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class DegreeCountVisitor extends PrintVisitor implements DownVisitor {

    /** Base of keys in configuration file */
    private static final String CFG_BASE = "DegreeCountVisitor.";
    /** Default filename */
    private static final String DEF_FILENAME = "tree-degree";

    /** Type of identifying data to print for each entry */
    protected String type = "data";
    /** Whether to report degrees only within a specific type of node */
    protected boolean intraType = false;


    /**
     * Default constructor.
     */
    public DegreeCountVisitor() {

	super( DEF_FILENAME );
    }
    
    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     */
    public DegreeCountVisitor( String filename ) {

	super( filename );
    }


    /**
     * Whether to visit after visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit after
     */   
    public boolean postfix() {

	return( true );
    }


    /**
     * Initialize the visitor state.
     *
     * @param <code>Tree</code> current tree instance
     * @return <code>boolean</code> whether to walk again
     */
    public boolean init( Tree tree ) {

	super.init( tree );

	// get type from config
	String t = cfgString( CFG_BASE + "type" );
	if( t != null )
	    type = t;
	// get intraType flag from config
	intraType = cfgBoolean( CFG_BASE + "intraType" );

	return( true );
    }


    /**
     * Print visitor's header info at start of file.
     *
     * @param <code>Tree</code> current tree instance
     */
    protected void printHeader( Tree tree ) {

	super.printHeader( tree );

	ps.println( "#" );
	ps.println( "# id\tdepth\tdownstream receivers\ttotal degree\tleaf degree\tinterior degree" );
    }

    /**
     * Print a single line of data for a node.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current tree node
     * @param <code>DegreeCount</code> current degree count
     */
    protected void printData( Tree tree, TreeNode node, DegreeCount count ) {

	// get identifying value to print for each entry
	Object id = node.data.get( type );
	if( id == null )
	    id = node.getIP();

	int depth = 0;
	try {
	    // figure out our current depth
	    depth = ((Integer)node.data.get( "depth" )).intValue();
	    
	} catch( Exception e ) {}
	
	// print out data, depth and degree of node
	String line = id.toString() + "\t" + depth + "\t" + count.toString();
	// add interior links
	line += "\t" + (count.links - count.leafs);
		
	Config.verbose( this, line );
	printData( tree, node, line );
    }


    /**
     * Visit the current node prior to visiting children.
     * This method is called if prefix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node ) {

	// visit only once
	DegreeCount count = null;
	if( ! markVisited( node ) ) {
	    // return basic count as a receiver (active or not?)
	    boolean active = eval.isActive( (Receiver)node );
	    count = new DegreeCount( active, node.data.get( type ) );
	}

	return( count );
    }

    /**
     * Visit the current node after visiting children.
     * This method is called if postfix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @param <code>Vector</code> list of return values passed back along the path
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node, Vector list ) throws VisitException {

	// get this node's data to match
	Object data = node.data.get( type );
	DegreeCount cnt = new DegreeCount( data );
	
	// mark node as seen and visited - only once
	if( markVisited( node ) )
	    return( cnt );

	// add children's degree counts to our own
	for( Enumeration enum = list.elements(); enum.hasMoreElements(); )
	    try {
		// get count from next child
		DegreeCount count = (DegreeCount)enum.nextElement();
		// add only same types if requested
		if( ! intraType || data == null || data.equals( count.data ) )
		    // add count to current total
		    cnt.add( count );
		else
		    // still track the number of downstream receivers
		    cnt.addReceivers( count );
		
	    } catch( NoSuchElementException nse ) {
	    } catch( Exception e ) {
		throw new VisitException( "error evaluating child activity" );
	    }
	
	// print out depth and degree of node
	printData( tree, node, cnt );

	return( cnt );
    }
}
