/* 
     File : GlobalVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : November 17, 2000
            1. Added support for reading defaults from config file.

  Content : A compund visitor which runs a pair of visitors, one walking up 
            the tree and a second which walks back down.
             
  $Id: GlobalVisitor.java,v 1.6 2001/08/22 17:44:21 robertc Exp $
*/

package mwalk.visitor;


import java.util.Vector;
import java.util.Enumeration;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;
import mwalk.eval.PrimaryPathEvaluator;
import mwalk.util.VisitException;


/**
 * A compund visitor which runs a pair of visitors, one walking up the tree 
 * and a second which walks back down.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class GlobalVisitor extends AbstractVisitor implements UpVisitor, DownVisitor {

    /** Base of keys in configuration file */
    private static final String CFG_BASE = "GlobalVisitor.";

    /** Visitor to use while walking up the tree */
    protected UpVisitor uv = null;
    /** Visitor to use while walking down the tree */
    protected DownVisitor dv = null;

    /** Flag indicating whether we are currently walking up the tree */
    protected boolean goingUp = false;

    /** Number of rounds and current repitition */
    protected long rounds = 0;
    /** Current round */
    protected long round = 0;


    /**
     * Default constructor.
     */
    public GlobalVisitor() {}

    public GlobalVisitor( int rounds ) {

	this.rounds = rounds;
    }

    public GlobalVisitor( DownVisitor dv ) {

	this.dv = dv;
    }

    public GlobalVisitor( UpVisitor uv, DownVisitor dv ) {

	this.uv = uv;
	this.dv = dv;
    }

    public GlobalVisitor( DownVisitor dv, int rounds ) {

	this.dv = dv;
	this.rounds = rounds;
    }

    public GlobalVisitor( UpVisitor uv, DownVisitor dv, int rounds ) {

	this.uv = uv;
	this.dv = dv;
	this.rounds = rounds;
    }


    /**
     * Whether to visit prior to visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit prior
     */
    public boolean prefix() {
	
	return( goingUp && uv.prefix() || ! goingUp && dv.prefix() );
    }


    /**
     * Whether to visit after visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit after
     */   
    public boolean postfix() {

	return( goingUp && uv.postfix() || ! goingUp && dv.postfix() );
    }


    /**
     * Whether to walk up the tree.
     *
     * @return <code>boolean</code> whether to walk up the tree
     */
    public boolean walkUp() {

	return( goingUp );
    }


    /**
     * Whether to walk down the tree.
     *
     * @return <code>boolean</code> whether to walk down the tree
     */
    public boolean walkDown() {

	return( ! goingUp );
    }


    /**
     * Initialize the visitor state.
     *
     * @param <code>Tree</code> current tree instance
     */
    public boolean init( Tree tree ) {

	// try to supply default values from config file
	if( uv == null )
	    uv = (UpVisitor)cfgVisitor( CFG_BASE + "upVisitor" );
	if( dv == null )
	    dv = (DownVisitor)cfgVisitor( CFG_BASE + "downVisitor" );
	if( rounds == 0 ) {
	    rounds = cfgInt( CFG_BASE + "rounds" );
	    if( rounds <= 0 )
		rounds = 1;
	}

	// ensure a down visitor was supplied
	if( dv == null ) {
	    System.err.println( "No DownVisitor was supplied!" );
	    return( false );
	}

	// initialize the up and down visitors
	boolean initDn, initUp = true;
	if( uv != null )
	    initUp = uv.init( tree );
	initDn = dv.init( tree );       
	    
	return( initUp && initDn && super.init( tree ) );
    }


    /**
     * Reset the visitor for a new pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     */
    public void reset( Tree tree ) {

	super.reset( tree );

	// reset the round count and prepare the up/down visitors
	round = 0;
	prepare( tree );
    }


    /**
     * Check whether the visitor wants another pass over the tree.
     * Repeat complete up/down series for total number of rounds.
     *
     * @param <code>Tree</code> vistited tree 
     * @return <code>boolean</code> whether to walk again
     */
    public boolean again( Tree tree ) {

	if( goingUp ) {
	    if( ! uv.again( tree ) ) {
		// we've gone up, now let's go down
		goingUp = false;
		dv.reset( tree );
	    }
	    
	} else if( ! dv.again( tree ) ) {
	    // repeat up/down series for a number of rounds
	    if( ++round < rounds )
		prepare( tree );
	    else
		return( false );
	}
	
	return( true );
    }


    /**
     * Prepare the up/down visitors for a new pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     */
    public void prepare( Tree tree ) {

	// reset direction flag
	goingUp = (uv != null);

	if( goingUp )
	    uv.reset( tree );
	else
	    dv.reset( tree );
    }


    /**
     * Check whether the next receiver should start a visiting path.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>Receiver</code> next receiver
     * @return <code>boolean</code> whether to visit next receiver
     */
    public boolean nextReceiver( Tree tree, Receiver recv ) {

	return( uv.nextReceiver( tree, recv ) );
    }


    /**
     * Get a list of active receivers.
     *
     * @param <code>Tree</code> current tree instance
     * @return <code>Enumeration</code> list of receivers to visit
     */
    public Enumeration getReceivers( Tree tree ) {

	return( uv.getReceivers( tree ) );
    }

    /**
     * Visit the current node prior to visiting parent.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitUp( Tree tree, TreeNode node ) throws VisitException {

	return( uv.visitUp( tree, node ) );
    }


    /**
     * Visit the current node after visiting parent.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @param <code>Object</code> list of return values passed back along the path
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitUp( Tree tree, TreeNode node, Vector list  ) throws VisitException {

	return( uv.visitUp( tree, node, list ) );
    }
    

    /**
     * Visit the current node prior to visiting children.
     * This method is called if prefix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node ) throws VisitException {
	
	return( dv.visitDown( tree, node ) );
    }
    
    /**
     * Visit the current node after visiting children.
     * This method is called if postfix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @param <code>Object</code> list of return values passed back along the path
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node, Vector list ) throws VisitException {
	
	return( dv.visitDown( tree, node, list ) );
    }


    /**
     * Get a list of parents for the current node applying correct activity
     * semantics.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>TreeNode[]</code> array of active parent nodes
     */
    public TreeNode[] getParents( Tree tree, TreeNode node ) {

	return( uv.getParents( tree, node ) );
    }


    /**
     * Get a list of children for the current node applying correct activity
     * semantics.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>TreeNode[]</code> list of active children nodes
     */
    public TreeNode[] getChildren( Tree tree, TreeNode node ) {

	return( dv.getChildren( tree, node ) );
	/*
	if( ! ((AbstractVisitor)dv).markSeen( node ) )
	    // get a list of all child links active by primary route
	    return( node.getActiveChildren( new PrimaryPathEvaluator() ) );
	else
	    return( new TreeNode[0] );
	*/
    }
}
