/* 
     File : PrimaryPathVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A visitor used by the builder to mark the primary path for 
            each receiver.

  $Id: PrimaryPathVisitor.java,v 1.5 2001/08/22 17:45:07 robertc Exp $
*/

package mwalk.visitor;


import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Link;


/**
 * A visitor used by the builder to mark the primary path for each receiver.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class PrimaryPathVisitor extends AbstractVisitor implements UpVisitor {

    /** A count of primary links in tree */
    protected int links = 0;
    /** A count of total egdes in graph */
    protected int edges = 0;


    /**
     * Default constructor.
     */
    public PrimaryPathVisitor() {}


    /**
     * Return the number of primary links in the tree.
     *
     * @return <code>int</code> number of primary links.
     */
    public int countLinks() {

	return( links );
    }

    /**
     * Return the total number of edges in the graph.
     *
     * @return <code>int</code> number of edges.
     */
    public int countEdges() {

	return( edges );
    }


    /**
     * Retrieve the list of parents of this node.
     *
     * @param <code>Tree</code> vistited tree 
     * @param <code>TreeNode</code> currently visited node
     * @return <code>TreeNode[]</code> array of parent nodes of currently visited node
     */
    public TreeNode[] getParents( Tree tree, TreeNode node ) {

	// get a list of available parents
	TreeNode parents[] = super.getParents( tree, node );
	Link plink = null;
	long duration = 0;
	
	// look for parent with longest duration
	for( int p = 0; p < parents.length; p++ ) {
	    // get the link from parent to node and calculate the duration bounded by the simulation 
	    Link link = parents[p].getChildLink( node );
	    long dur = link.activity.getDuration( tree.getStop() );
	    if( dur >= duration ) {
		// save the longer duration
		duration = dur;
		// if we have a reference to another link, mark it as not being the longest
		if( plink != null )
		    plink.primary = false;
		// save this link as the primary
		plink = link;
	    } else
		link.primary = false;
	}

	// increment the egde count by number of parent links
	edges += parents.length;
	
	// mark the primary link and increment link count
	if( plink != null ) {
	    plink.primary = true;
	    links++;
	}       

	// return parent nodes
	return( parents );
    }
}
