/* 
     File : VisualVisitor.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A visitor that outputs a data set that can be used 
            to visualize the tree.
             
  $Id: VisualVisitor.java,v 1.5 2001/08/22 17:45:08 robertc Exp $
*/

package mwalk.visitor;


import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.io.*;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Source;
import mwalk.core.Config;


/**
 * A visitor that outputs a data set that can be used
 * to visualize the tree.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class VisualVisitor extends PrintVisitor implements DownVisitor {

    /** Base of keys in configuration file */
    private static final String CFG_BASE = "VisualVisitor.";
    /** Default file name if none supplied */
    private static final String DEF_FILENAME = "tree-visual";

    protected long times[] = null;
    protected int time = 0;
    protected boolean showIP = false;


    /**
     * Default constructor.
     */ 
    public VisualVisitor() {
	
	super( DEF_FILENAME );

	times = new long[] { 0 };
	evaluator( new mwalk.eval.RandomEvaluator() );
    }

    public VisualVisitor( long times[] ) {

	this.times = times;
    }


    /**
     * Whether to visit prior to visiting parent/children.
     *
     * @return <code>boolean</code> whether to visit prior
     */
    public boolean prefix() {

	return( true );
    }

    /**
     * Initialize the visitor state.
     *
     * @param <code>Tree</code> current tree instance
     * @return <code>boolean</code> whether to walk again
     */
    public boolean init( Tree tree ) {

	if( super.init( tree ) ) {
	    if( times == null ) {
		Vector timeList = new Vector( 3, 3 );
		try {
		    BufferedReader in = new BufferedReader( new InputStreamReader( new FileInputStream( "times" ) ) );
		    String line;
		    
		    // parse the list of times to visit
		    while( (line = in.readLine()) != null )
			timeList.add( new Long( line ) );
		    
		} catch( Exception e ) {}
		
		try {
		    // convert list to an array
		    times = new long[timeList.size()];
		    for( int i = 0; i < times.length; i++ )
			times[i] = ((Long)timeList.get( i )).longValue();
		
		} catch( Exception e ) {
		    Config.verbose( "VisualVisitor: error filling time list from file\n\t" + e.getMessage() );
		    return( false );
		}
	    }

	    mwalk.util.ReceiverRandomizer rand = new  mwalk.util.ReceiverRandomizer( tree.receivers() );
	    rand.randomize( tree );

	    return( times.length > 0 );
	} else
	    return( false );
    }

    /**
     * Reset the visitor for a new pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     */
    public void reset( Tree tree ) {

	super.reset( tree );
    }

    /**
     * Check whether the visitor wants another pass over the tree.
     *
     * @param <code>Tree</code> vistited tree 
     * @return <code>boolean</code> whether to walk again
     */
    public boolean again( Tree tree ) {

	return( super.again( tree ) && ++time < times.length );
    }


    /**
     * Visit the current node prior to visiting children.
     * This method is called if prefix() returns true.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node ) {

	// mark node as seen and visited
	if( ! haveSeen( node ) ) {
	    StringBuffer buf = new StringBuffer( 50 );

	    buf.append( "NODE [\n\tNAME IP_" + node.getIP() + "\n" );
	    
	    TreeNode[] kids = node.getActiveChildren( eval, times[time] );
	    if( kids.length > 0 ) {
		buf.append( "\tCHILDREN [\n" );
		for( int k =0; k < kids.length; k++ ) 
		    buf.append( "\t\tIP_" + kids[k].getIP() + "\n" );
		buf.append( "\t]\n" );
	    }
	    
	    buf.append( "]\n" );
	    printData( tree, buf.toString() );
	}
	
	return( null );
    }

    /**
     * Get a list of children for the current node applying correct activity
     * semantics.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>TreeNode[]</code> list of active children nodes
     */
    public TreeNode[] getChildren( Tree tree, TreeNode node ) {

	// check if this node has already been seen
	if( ! markSeen( node ) )
	    // get all children on active links for the current time
	    return( node.getActiveChildren( eval, times[time] ) );
	else
	    return( new TreeNode[0] );
    }
}
