#! /usr/bin/env perl
#
#     File : upath
#   Author : Robert Chalmers
#
# Original : January 6, 2000
#  Revised :
#
#  Content : main script for gathering unicast path lengths
#

package main;

# import external classes/modules
use FileHandle;

# import internal classes/modules
use Mwalk::Args;
use Mwalk::Receiver;
use Mwalk::TraceSet;
use Mwalk::TraceRoute;


# enforce strictness
use strict qw( vars refs subs );

# constant pattern for ip address
my $IP = '(?:\d{1,3}\.){3}\d{1,3}';

# command-line arguments used to control execution
my $args = new Mwalk::Args( 1 );
# stats kept for each type of data (rtcp, mtrace, interval)
my %stats = ( counts => { with => 0, tot => 0, skip => 0 },
              skips  => {form => 0, unreachable => 0, traceless => 0}
            );
# list of processed receivers hashed by ip address
my %receivers;


# procedure declarations
use subs qw(printUsage parseArgs initLogs createReceivers saveStats saveData goInteractive);


# parse command-line options
parseArgs;
# prepare the output logs
initLogs;

# mark the start time for processing
$args->setStart( time );

# create traceroute information for listed receivers
createReceivers;

# mark the stop time for processing
$args->setStop( time );

# output if indicated
saveData;
saveStats if $args->{statFile};

# provide an interactive command line if requested
goInteractive if $args->{interactive};

exit;



#
# print script usage
#
# params
#   optional exit code
#
sub printUsage {

  my $code = shift;
  # default exit code to -1
  $code = -1 unless defined( $code );

  print <<EOU;
mwalk upath [-i|--interactive] [-v|--verbose] [-l|--log]
            [-so|--source=<source>] [-r|--receiver=<rfile>]
            [-c|--config=<cfg>] [-s|--stats=<sfile>] [-d|--dir=<outdir>]
            [-f|--fussy] [-a|--append] [-b|--build] [-h|--help]
Options:
  -i  --interactive  run in an interactive mode
  -v  --verbose      print verbose messages
  -l  --log          generate processing logs
  -so --source       host to use as source of traceroute [localhost]
  -r  --receiver     list of receivers to trace [receivers]
  -c  --config       configuration file []
  -s  --stats        generate statistics log [stats]
  -d  --dir          output directory [.]
  -f  --fussy        use strict parsing
  -a  --append       append to existing log files
  -b  --build        accumulate traces in memory
  -h  --help         print this message
EOU

  # exit with provided code
  exit $code;
}

   
#
# parse the command line options
#
sub parseArgs {

  # process the command line options
  $args->parse( "interactive!", "verbose!", "output_gen|log!",
		"append!", "build|b!", "fussy!", "help", 
		"source=s", "receiver=s", "stats|s=s", 
		"dir=s", "config=s" );

  # sanity check
  $args->{interactive} or -r $args->{recvFile}
                       or print "\nMust specify an input file (-r)\n" and printUsage;
}


#
# prepare the output logs if logging was indicated
#
sub initLogs {

  # return if logging was not requested
  return unless $args->{genOutput};

  my $mode = ($args->{append}) ? ">>" : ">";

  # open log files or quit
  $args->{utraceHandle} = new FileHandle( "$mode $args->{utraceFile}" ) 
                         or die "Couldn't open TraceRoute log $args->{utraceFile}: $!\n";

  # print log headers
  print { $args->{utraceHandle} } "# UNICAST PATH LOG - generated by mwalk\n";
  print { $args->{utraceHandle} } "# Started at ", scalar localtime, "\n";

  # clear any temporary logs
  my $set = new Mwalk::TraceSet( $args );
  $set->clearLogs();
}


#
# add a new receiver if that receiver is reachable
# 
# params
#   ip address of receiver
#
# return
#   whether the receiver was added
# 
sub createReceiver {

  my $ip = shift;

  # ensure receiver not already in list
  unless( exists $receivers{$ip} ) {
    # check whether the receiver is reachable
    if( system( "$args->{ping} $args->{pingFlags} $ip > /dev/null" ) == 0 ) {
      print "Adding receiver for $ip\n" if $args->{verbose};
      # create a new receiver
      $receivers{$ip} = new Mwalk::Receiver( $ip, $args );
      return 1;
    } else {
      print "Receiver $ip unreachable\n" if $args->{verbose};
      $stats{skips}{unreachable}++;      
    }
  }

  return 0;
}


#
# generate an traceroute for this receiver
#
# params
#   ip address of receiver
#
sub createTrace {  

  my $ip = shift;
  my ($trace, $set) = (undef, new Mwalk::TraceSet( $args ));
  
  print "Starting TraceRoute processing for $ip\n" if $args->{verbose};
    
  # generate and parse a traceroute for the given receiver
  $trace = $set->generate( $ip, "utrace" );

  # check if parsed correctly, add to receiver or remove receiver entirely
  if( defined( $trace ) ) {
    $receivers{$ip}->add( $trace );
  } else {
    delete $receivers{$ip};
    $stats{skips}{traceless}++;
    $stats{count}{skip}++;
  } 
}
  

#
# take a list of receivers and try to create traceroutes for each
#
sub createReceivers {

  my ($fh, $line, $ip);
  my $reps = 1;

  # open receivers file
  $fh = new FileHandle( "< $args->{recvFile}" ) or die "Couldn't open receivers file: $!\n";

  print "Starting receiver processing for $args->{recvFile}\n" if $args->{verbose};
  print "Enter number of reps:" and $reps = <STDIN> + 0 if $args->{interactive};

  # parse each line as a receiver's ip address
  while( $line = <$fh> and $reps ) {
    # ignore comments
    next if $line =~ m/^\s*\#/o;

    # look for ip address in line
    if( ($ip) = ($line =~ m/($IP)/o) ) {
      # try to create a new receiver with traceroute data
      createTrace( $ip ) if createReceiver( $ip );
    } else {
      print "Didn't find ip address on line\n";
      $stats{skips}{form}++;
    }

  } continue {
    print "Enter number of reps:" and $reps = <STDIN> + 0 if $args->{interactive} and ! --$reps;
  }

  # clear any temporary logs
  my $set = new Mwalk::TraceSet( $args );
  $set->clearLogs();
}


#
# save statistics about data collection
#
sub saveStats {
 
  my $recvs = 0;

  print "Compiling statistics\n" if $args->{verbose};

  # compute stats for each receiver
  my ($ip, $recv);
  while( ($ip, $recv) = each %receivers ) {
    # compute stats for traceroute data
    foreach my $data ("utrace") {
      my $len = $recv->size( $data );
      $stats{counts}{with}++ if $len;
      $stats{counts}{tot} += $len;
    }
    # count the number of receivers
    $recvs++;
  }

  # open a new file for writing if stat file specified
  my $fh = new FileHandle( (($args->{append}) ? ">>" : ">") . " $args->{statFile}" ) 
           or die "Can't open stat file $args->{statFile}: $!\n";
  
  # print runtime arguments
  $args->save( $fh );
  # print runtime times
  print $fh "start:", scalar localtime $args->{start}, "\n";
  print $fh "stop:", scalar localtime $args->{stop}, "\n";
  print $fh "elapsed:", $args->{stop} - $args->{start}, " sec.\n\n";

  # print stats
  print $fh "STATS:\n";
  print $fh "receivers:$recvs\n";
  print $fh "SKIPS:\n";
  foreach my $skip (keys %{$stats{skips}}) {
    print $fh "$skip:$stats{skips}{$skip}\n";
  }
  print $fh "with:$stats{counts}{with}\n";
  print $fh "tot:$stats{counts}{tot}\n";
  print $fh "skip:$stats{counts}{skip}\n";

  # close the stats file
  close $fh;
}


#
# save collected data into output file(s)
#
sub saveData {

  my ($fh, $mode);
  
  # open tabulation files to record which receivers have mtraces
  $mode = ($args->{append}) ? ">>" : ">";
  $fh = new FileHandle( "$mode $args->{dir}receivers.upath" ) or die "Couldn't open upath receivers file: $!\n";

  # save data for each receiver
  my ($ip, $recv);
  while( ($ip, $recv) = each %receivers ) {     
    print "Saving data for receiver $ip\n" if $args->{verbose};
    print $fh $recv->ip() . "\t" . $recv->upath(), "\n" if $recv->upath() > 0;
  }

  close $fh;
}



#
# enter an interactive mode to allow on-line querying
#
sub goInteractive {

  my ($line, $cmd) = (undef, "");

  print "Entering interactive mode\n" if $args->{verbose};
  # print prompt
  print "\nupath> ";

  # loop around until user exits
  while( $line = <STDIN> ) {
    # append the line to the command
    $cmd .= $line;
    
    # check for line continuation
    print "> " and next if $line =~ m/\\\w*$/;
    # check for exit command
    last if $cmd =~ m/^\w*exit\w*(\\\w*)*$/;
    
    # evaluate command if something passed
    print "error: $@\n" if $cmd !~ m/^\w*$/ and ! defined( eval $cmd );

    # clear command and reprint prompt
    $cmd = "";
    print "upath> ";
  }
}
