#
#     File : Receiver.pm
#   Author : Robert Chalmers
#
# Original : November 22, 1999
#  Revised :
#
#  Content : class module encapsulating an individual multicast receiver
#

package Mwalk::Receiver;


# import external modules
use FileHandle;

# import internal classes/modules
use Mwalk::Args;


# enforce strictness
use strict qw( vars refs subs );

# member variable structure
my %members = ( ip        => "",
		firstTime => 0,
		lastTime  => 0,
		intervals => 0,
		interval  => undef,
		byes      => 0,
		patches   => 0,
		completes => 0,
		rtcps     => 0,
		rtcp      => undef,
		mtraces   => 0,
		mtrace    => undef,
		utraces   => 0,
		utrace    => undef,
		upath     => 0,
		args      => undef
	      );


#
# constructor
#
# params
#   class name or object reference
#   ip address for this receiver
#   configuration arguments
#
# return
#   new object reference
#
sub new {

  my ($that, $ip, $args) = @_;
  my $class = ref( $that ) || $that;

  # build new object from template
  my $self = { %members };
  bless $self, $class;

  #initialize ip address and arguments
  $self->{ip} = $ip;
  $self->{args} = $args;
  # initialize anonymous arrays and hashes
  $self->{interval} = [];
  $self->{rtcp} = [];
  $self->{mtrace} = [];
  $self->{utrace} = [];

  # clear existing data files for this receiver
  $self->clearOutput() unless $args->append();

  return $self;
}



#
# return the size of one of the receiver's lists
#
# params
#   which list to check (interval, rtcp, mtrace, utrace)
#
# return
#   length of requested list or undefined if invalid list
#
sub size {

  my ($self, $list) = (shift, lc( shift ));

  # return the count on the requested list
  return $self->{"${list}s"};
}



#
# add a new data object to the receiver's list
#
# params
#   reference to an RTCP or MTrace object
#
# return
#   whether the object was added successfully
#
sub add {

  my ($self, $data) = @_;

  # check if object is correct for this list
  return 0 unless $data->receiver() eq $self->{ip};

  # handle special-case statistics
  if( ref( $data ) eq "Mwalk::RTCP" ) {
    $self->addRTCP( $data );
  } elsif( ref( $data ) eq "Mwalk::MTrace" ) {
    $self->addMTrace( $data );
  } elsif( ref( $data ) eq "Mwalk::TraceRoute" ) {
    $self->addUTrace( $data );
  } else {
    die "Asked to add an invalid component to a receiver: ", ref( $data ); 
  }
}


#
# add a new rtcp object to the receiver's list
#
# params
#   reference to an RTCP object
#
sub addRTCP {

  my ($self, $rtcp) = @_;
  my $timestamp = $rtcp->timestamp();

  # increment count of objects
  $self->{rtcps}++;

  # add to receiver statistics
  $self->{byes}++ if $rtcp->bye();
  $self->{firstTime} = $timestamp if ! $self->{firstTime} or $timestamp < $self->{firstTime};
  $self->{lastTime} = $timestamp if $timestamp > $self->{lastTime};

  # find last interval
  my $interval = @{$self->{interval}} - 1;
  if( $interval < 0 or $self->{interval}[$interval]{bye} or $timestamp - $self->{interval}[$interval]{stop} > $self->{args}{timeout} ) {
    # add new interval if empty, explicit bye, or over our timeout explicit bye
    push @{$self->{interval}}, {start => $timestamp, stop => $timestamp, count => 1, bye => $rtcp->bye()};
    $self->{intervals}++;
  } else {
    # otherwise, extend stop time of interval
    $self->{interval}[$interval]{stop} = $timestamp;
    $self->{interval}[$interval]{count}++;
    $self->{interval}[$interval]{bye} = $rtcp->bye();
  }

  # push object onto list if building in-memory
  push @{$self->{rtcp}}, $rtcp if $self->{args}{build};
  # save parsed rtcp object if requested
  $self->saveData( undef, $rtcp ) if $self->{args}{rtcpOutput};
}


#
# add a new mtrace object to the receiver's list
#
# params
#   reference to an MTrace object
#
sub addMTrace {

  my ($self, $mtrace) = @_;

  # increment object count
  $self->{mtraces}++;

  # add to receiver statistics
  $self->{patches}++ if $mtrace->patch();
  $self->{completes}++ if $mtrace->complete();
  $self->{firstTime} = $mtrace->start() if ! $self->{firstTime} or $mtrace->start() < $self->{firstTime};
  $self->{lastTime} = $mtrace->start() if $mtrace->start() > $self->{lastTime};

  # push object onto list if building in-memory
  push @{$self->{mtrace}}, $mtrace if $self->{args}{build};
  # save parsed mtrace object if requested
  $self->saveData( undef, $mtrace ) if $self->{args}{mtraceOutput};
}


#
# add a new traceroute object to the receiver's list
#
# params
#   reference to an TraceRoute object
#
sub addUTrace {

  my ($self, $utrace) = @_;

  if( $utrace->isComplete() ) {
    # increment object count
    $self->{utraces}++;

    # save minimum path length
    if( $self->{upath} == 0 or $self->{upath} > @{$utrace->hops()} ) {
      $self->{upath} = scalar @{$utrace->hops()};
    }

    # push object onto list if building in-memory
    push @{$self->{utrace}}, $utrace if $self->{args}{build};
    # save parsed traceroute object if requested
    $self->saveData( undef, $utrace ) if $self->{args}{utraceOutput};
  }
}



#
# write the complete receiver data out to file
#
# params
#   self reference
#   optional open file handle
#
sub save {

  my ($self, $file) = @_;

  # print receiver data and intervals
  $self->saveReceiver( $file );
  $self->saveInterval( $file ) if $self->{intervals};

  # save rtcp and mtrace data if requested
  $self->saveData( $file, "rtcp" ) if $self->{args}{rtcpOutput} and $self->{rtcps};
  $self->saveData( $file, "mtrace" ) if $self->{args}{mtraceOutput} and $self->{mtraces};
  $self->saveData( $file, "utrace" ) if $self->{args}{utraceOutput} and $self->{utraces};
}


#
# write the receiver specific data out to file
#
# params
#   self reference
#   optional open file handle
#
sub saveReceiver {

  my ($self, $file) = @_;
  my $fh;

  # open the save file unless handle was provided
  if( ref( $file ) ) {
    $fh = $file; 
  } else {
    $file = "$self->{args}{dir}$self->{ip}.receiver";
    $fh = new FileHandle( ">> $file" ) or die "Can't open save file $file: $!\n";
  }
 
  # print receiver data
  print $fh "RECEIVER:$self->{ip}\n";
  print $fh "first:$self->{firstTime}\nlast:$self->{lastTime}\n";
  print $fh "byes:$self->{byes}\npatches:$self->{patches}\ncompletes:$self->{completes}\n";
  print $fh "intervals:$self->{intervals}\nrtcps:$self->{rtcps}\n";
  print $fh "mtraces:$self->{mtraces}\nutraces:$self->{utraces}\n";

  # close the file if we opened it
  close $fh unless ref( $file );
}


#
#
# save all or a single interval
#
# params
#   self reference
#   optional open file handle
#   optional interval to save
#
sub saveInterval {

  my ($self, $file, $interval) = @_;
  my $fh;

  # open the save file unless handle was provided
  if( ref( $file ) ) {
    $fh = $file; 
  } else {
    $file = "$self->{args}{dir}$self->{ip}.interval";
    $fh = new FileHandle( ">> $file" ) or die "Can't open save file $file: $!\n";
  }

  # save a single interval or entire set
  if( ref( $interval ) ) {
    # add in the timeout value if stop wasn't explicit
    $interval->{stop} += $self->{args}{timeout} unless $interval->{bye};
    print $fh "$interval->{start}\t$interval->{stop}\t$interval->{count}\t";
    print $fh ($interval->{bye}) ? "B" : "T", "\n";
  } else {
    foreach $interval (@{$self->{interval}}) {
      $self->saveInterval( $fh, $interval );
    }
  }

  # close the file if we opened it
  close $fh unless ref( $file );
}


#
# write the rtcp, mtrace, or utrace data out to file
#
# params
#   self reference
#   optional open file handle
#   name of data set or actual data object to save
#
sub saveData {

  my ($self, $file, $data) = @_;
  my ($fh, $type);

  # figure out which data set we're dealing with
  if( ref( $data ) ) {
    # use stripped down class name
    $type = lc( ref( $data ) );
    $type =~ s/[^:]*:://go;
  } else {
    $type = $data;
  }

  # open the save file unless handle was provided
  if( ref( $file ) ) {
    $fh = $file; 
  } else {
    $file = "$self->{args}{dir}$self->{ip}.$type";
    $fh = new FileHandle( ">> $file" ) or die "Can't open save file $file: $!\n";
  }


  # print individual item or all from that data set
  if( ref( $data ) ) {
    $data->save( $fh );
  } else {
    foreach my $item (@{$self->{$data}}) {
      $item->save( $fh );
    }
  }

  # close the file if we opened it
  close $fh unless ref( $file );
}



#
# open save files for this receiver
#
# params
#   self reference
#
sub clearOutput {

  my $self = shift;

  foreach my $suffix ("receiver", "interval", "rtcp", "mtrace", "traceroute") {
    unlink "$self->{args}{dir}$self->{ip}.$suffix";
  }  
}



#
# class destructor
#
# params
#   self reference
#
sub DESTROY {

  my $self = shift;

  #print "Destroying Receiver for $self->{ip}.\n";
}



#
# provide access to member variables as automatic methods
#
# params
#   self reference
#
sub AUTOLOAD {

  my $self = shift;
  my $type = ref( $self ) || die "Access method called without object reference: $!";
  my $name = $Mwalk::Receiver::AUTOLOAD;

  # strip fully-qualified protion of method
  $name =~ s/.*://;
  # ensure member exists
  exists $self->{$name} or die "Access method called on non-existent member: $name";

  # set or get member variable
  if( @_ ) {
    return $self->{$name} = shift;
  } else {
    return $self->{$name};
  }
}
