/* 
     File : Link.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A class representing a link between a parent and a child node.

  $Id: Link.java,v 1.5 2001/10/19 18:58:25 robertc Exp $             
*/

package mwalk.core;


import java.util.Hashtable;
import java.io.Serializable;
import java.io.ObjectInputStream;
import java.io.IOException;


/**
 * A class representing a link between a parent and a child node.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class Link implements Serializable {

    /** Serial version ID */
    static final long serialVersionUID = -4932757547705029426L;

    /** Child node on other end of link */
    public TreeNode node;
    /** Table of active periods for this link */
    public PeriodTable activity = new PeriodTable();

    /** Flag indicating primary links */
    public transient boolean primary = false;
    /** Flag indicating time-independent link activity */
    public transient boolean active = false;
    /** Storage available to visitors to store partial results */
    public transient Hashtable data = new Hashtable();


    /**
     * Constructor.
     *
     * @param <code>TreeNode</code> node on other end of link
     */
    public Link( TreeNode node ) {

	this.node = node;
    }


    /**
     * Check whether link is marked primary.
     *
     * @return <code>boolean</code> whether link is primary
     */
    public boolean isPrimary() {

	return( primary );
    }


    /**
     * Check whether link is explicitly marked active.
     *
     * @return <code>boolean</code> whether link is active
     */
    public boolean isActive() {

	return( active );
    }


    /**
     * Check whether the link is active for a given timestamp.
     * 
     * @param <code>long</code> session timestamp
     */
    public boolean isActive( long time ) {

	return( activity.check( time ) );
    }


    /**
     * Mark the link as active for a duration between two timestamps.
     * 
     * @param <code>long</code> session timestamp
     */
    public long activate( long time, long next ) {

	return( activity.add( time, next ) );
    }


    /**
     * Mark the link as inactive starting at the passed timestamp.
     * 
     * @param <code>long</code> session timestamp
     */
    public long deactivate( long time ) {

	return( activity.bound( time ) );
    }


    /**
     * Clear activity table and stored data.
     */
    public void clearData() {

	// clear data in the activity table
	activity.clearData();
	// clear local data
	data.clear();
    }


    /**
     * Initialize transient variables after reading the object in.
     *
     * @param <code>ObjectInputStream</code> input stream
     * @exception <code>IOException</code> if object not read correctly
     */
    private void readObject( ObjectInputStream in ) throws IOException, ClassNotFoundException {
	
	// handle default read
	in.defaultReadObject();

	// initialize transient variables
	data = new Hashtable();
    }
}
