/* 
     File : PeriodEntry.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : A class representing a single period of activity for a link 
            or receiver.
   
  $Id: PeriodEntry.java,v 1.4 2001/10/19 18:58:25 robertc Exp $          
*/

package mwalk.core;


import java.util.Hashtable;
import java.io.Serializable;
import java.io.ObjectInputStream;
import java.io.IOException;


/**
 * A class representing a single period of activity for a link or receiver.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class PeriodEntry implements Serializable {

    /** Serialized verios ID */
    static final long serialVersionUID = 9136775705866627006L;

    /** Starting timestamp */
    public long start = 0;
    /** Ending timestamp */
    public long stop = Long.MAX_VALUE;
    /** Optional data which can be stored with period */
    public transient Hashtable data = new Hashtable();


    /** 
     * Default constructor.
     */
    public PeriodEntry() {}

    /**
     * Constructor.
     *
     * @param <code>long</code> starting timestamp
     */
    public PeriodEntry( long start ) {

	this.start = start;
    }

    /**
     * Constructor.
     *
     * @param <code>long</code> starting timestamp
     * @param <code>long</code> ending timestamp
     */
    public PeriodEntry( long start, long stop ) {

	this.start = start;
	this.stop = stop;
    }   

    /**
     * Copy constructor.
     *
     * @param <code>PeriodEntry</code> entry to copt
     */
    public PeriodEntry( PeriodEntry other ) {

	start = other.start;
	stop = other.stop;
    }

    
    /**
     * Determine whether the given timestamp falls within this period.
     *
     * @param <code>long</code> timestamp to check
     * @retun <code>boolean</code> whether period contains timestamp
     */
    public boolean contains( long time ) {

	return( time >= start && time < stop );
    }


    /**
     * Calculate duration of period in terms of timestamp units.
     *
     * @return <code>long</code> duration of period
     */
    public long duration() {
	
	if( stop < Long.MAX_VALUE )
	    return( Math.max( stop - start, 0 ) );
	else
	    return( Long.MAX_VALUE );
    }


    /**
     * Calculate duration of period from the start to the given bound in 
     * terms of timestamp units.
     *
     * @return <code>long</code> duration of bounded period
     */
    public long duration( long bound ) {

	if( bound < stop )
	    return( Math.max( bound - start, 0 ) );
	else
	    return( duration() );
    }

    /**
     * Clear temporary data for entry
     */
    public void clearData() {

	// clear local data
	data.clear();
    }


    /**
     * Initialize transient variables after reading the object in.
     *
     * @param <code>ObjectInputStream</code> input stream
     * @exception <code>IOException</code> if object not read correctly
     */
    private void readObject( ObjectInputStream in ) throws IOException, ClassNotFoundException {
	
	// handle default read
	in.defaultReadObject();

	// initialize transient variables
	data = new Hashtable();
    }
}
