/* 
     File : RandomRandomizer.java
   Author : Robert Chalmers

 Original : December 1999
  Revised : 

  Content : Class which randomly selects a number of receivers as being active.
         
  $Id: ReceiverRandomizer.java,v 1.6 2002/03/29 18:06:22 robertc Exp $    
*/

package mwalk.util;


import java.util.Vector;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Receiver;


/**
 * Class which randomly selects a number of receivers as being active.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class ReceiverRandomizer {

    /** List of receivers in tree */
    protected TreeNode[] recvs = null;

    /** Number of receivers to mark as active */
    protected int active = 1;
    /** Maximum number of receivers available */
    public int max = 1;


    /**
     * Default constructor.
     */
    public ReceiverRandomizer() {}

    /**
     * Constructor.
     *
     * @param <code>int</code> number of active receivers
     */
    public ReceiverRandomizer( int active ) {
	
	this.active = active;
    }


    /**
     * Retrieve current number of active receivers.
     *
     * @return <code>int</code> number of active receivers
     */
    public int getActive() {

	return( active );
    }

    /**
     * Increment current number of active receivers.
     *
     * @return <code>int</code> number of active receivers
     */
    public int incActive() {

	return( ++active );
    }

    /**
     * Set current number of active receivers.
     *
     * @param <code>int</code> number of active receivers
     */
    public void setActive( int active ) {

	this.active = active;
    }

    /**
     * Retrieve maximum number of active receivers.
     *
     * @return <code>int</code> maximum number of active receivers
     */
    public int maxActive() {

	return( max );
    }


    /**
     * Clear all active receivers.
     *
     * @param <code>Tree</code> current tree
     */
    public void reset( Tree tree ) {

	// get receiver list as an array
	if( recvs == null ) 
	    recvs = TreeNode.toNodeArray( tree.getReceivers(), tree.receivers() );

	// set all receivers to non-active
	for( int r = 0; r < recvs.length; r++ )
	    ((Receiver)recvs[r]).active = false;
    }

    /**
     * Randomly activate the appropriate number of receivers.
     *
     * @param <code>Tree</code> current tree
     */
    public void randomize( Tree tree ) {

	// clear any current selections
	reset( tree );

	// create a list of available receivers
	Vector avail = TreeNode.toNodeVector( tree.getReceivers(), tree.receivers() );
	try {
	    // set the maximum number of available receivers
	    max = avail.size();
   
	    // randomly select active receivers
	    for( int i = 0; i < active && i < max; i++ ) {
		// compute a random index into list of availbale receivers
		int r = (int)(Math.random() * avail.size());
		// remove selected receiver and mark as active
		Receiver recv = (Receiver)avail.remove( r );
		recv.active = true;
	    }
	} catch( ArrayIndexOutOfBoundsException obe ) {}
    }
}
