/* 
     File : DataVisitor.java
   Author : Robert Chalmers

 Original : October 17, 2001
  Revised : 

  Content : A visitor that assigns data elements to each node in the tree from
            an ecternal file.  The file should be formatted with each line
	    containing a node's adress and the associated data separated by a 
	    space.
             
  $Id: DataVisitor.java,v 1.3 2002/05/10 18:07:57 robertc Exp $
*/

package mwalk.visitor;


import java.util.Properties;
import java.io.*;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Config;


/**
 * A visitor that assigns data elements to each node in the tree from
 * an ecternal file.  The file should be formatted with each line
 * containing a node's adress and the associated data separated by a
 * space.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class DataVisitor extends AbstractVisitor implements DownVisitor {

    /** Base of keys in configuration file */
    private static final String CFG_BASE = "DataVisitor.";

    /** Filename to use for data */
    protected String filename = null;
    /** Default filename if none supplied */
    protected String def_filename = "tree-data";

    /** Type of data - used as node's hash key */
    protected String type = "data";
    /** List of data items keyed by the node's address */
    protected Properties data = new Properties();


    /**
     * Default constructor.
     */
    public DataVisitor() {}

    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     */
    public DataVisitor( String filename ) {
	
	def_filename = filename;
    }

    
    /**
     * Retrieve the current filename.
     *
     * @return <code>String</code> current filename
     */    
    public String getFilename() {

	return( filename );
    }

    /**
     * Set the current filename prior to init().
     *
     * @param <code>String</code> new filename
     */
    public void setFilename( String filename ) {

	this.filename = filename;
    }


    /*
     * Visit the current node prior to visiting its children.
     *
     * @return <code>boolean</code> visit node first
     */
    public boolean prefix() {

	return( true );
    }


    /**
     * Initialize visitor.
     *
     * @param <code>Tree</code> current tree instance
     */
    public boolean init( Tree tree ) {

	super.init( tree );

	// get filename from config
	filename = cfgString( CFG_BASE + "filename" );
	if( filename == null )
	    filename = def_filename;
	// get type from config
	String t = cfgString( CFG_BASE + "type" );
	if( t != null )
	    type = t;

	// read data as a set of properties
	String visitor = getClass().getName();
	try {
	    data.load( new FileInputStream( getFilename() ) );
	    
	} catch( Exception e ) {
	    Config.verbose( visitor + ".init(): error reading data file" );
	    return( false );
	}

	// setup a primary tree evaluator
	evaluator( new mwalk.eval.PrimaryTreeEvaluator() );

	return( true );
    }


    /**
     * Visit the current node prior to visiting children.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node ) {

	// mark node as seen and visited
	if( ! haveSeen( node ) ) {
	    // lookup data for this node
	    Object val = data.getProperty( node.getIP(), "" );

	    // save data with node
	    Config.verbose( this, "Assigned " + type + "=" + val.toString() );
	    node.data.put( type, val );
	}
	 	
	return( null );
    }
}
