/* 
     File : ExportVisitor.java
   Author : Robert Chalmers

 Original : November 17, 2000
  Revised : 

  Content : A visitor that exports an ascii representation of the current tree.
             
  $Id: ExportVisitor.java,v 1.5 2002/05/10 18:07:57 robertc Exp $
*/

package mwalk.visitor;


import java.util.Vector;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.io.*;

import mwalk.core.Tree;
import mwalk.core.TreeNode;
import mwalk.core.Source;
import mwalk.core.Config;
import mwalk.util.VisitException;


/**
 * A visitor that exports an ascii representation of the current tree.
 *
 * @author Robert Chalmers
 * @version 1.0 
 */
public class ExportVisitor extends PeriodicVisitor implements DownVisitor {

    /** Base of keys in configuration file */
    private static final String CFG_BASE = "ExportVisitor.";
    /** Default file name if none supplied */
    private static final String DEF_FILENAME = "tree-export";

    /** Current period that we are exporting */
    protected long period = 1;
    /** Whether to export true IP address or whether to mangle it */
    protected boolean showIP = false;
    /** Integer count used to assign virtual IDs to nodes */
    protected int nextID = 0;


    /**
     * Default constructor.
     */
    public ExportVisitor() {
	
	super( DEF_FILENAME );
    }

    /**
     * Constructor.
     *
     * @param <code>String</code> export filename
     */
    public ExportVisitor( String filename ) {
	
	super( filename );
    }
    
    /**
     * Constructor.
     *
     * @param <code>long</code> number of periods to cover session
     */
    public ExportVisitor( long periods ) {

	super( DEF_FILENAME, periods );
    }

    /**
     * Constructor.
     *
     * @param <code>String</code> new default filename
     * @param <code>long</code> number of periods to cover session
     */
    public ExportVisitor( String filename, long periods ) {

	super( filename, periods );
    }


    /*
     * Visit the current node prior to visiting its children.
     *
     * @return <code>boolean</code> visit node first
     */
    public boolean prefix() {

	return( true );
    }

    /**
     * Initialize visitor.
     *
     * @param <code>Tree</code> current tree instance
     */
    public boolean init( Tree tree ) {

	// allow config to override showIP
 	if( cfgBoolean( CFG_BASE + "showIP" ) )
	    showIP = true;

	// handle base initialization
	boolean ok = super.init( tree );

	// walk entire tree if just one period specified
	if( periods == 1 )
	    evaluator( new mwalk.eval.PrimaryTreeEvaluator() );

	return( ok );
    }

    /**
     * Determine whether to walk the tree again.
     *
     * @param <code>Tree</code> current tree instance
     * @return <code>boolean</code> whether to walk again
     */
    public boolean again( Tree tree ) {

	return( super.again( tree ) );
    }


    /**
     * Visit the current node prior to visiting children.
     *
     * @param <code>Tree</code> current tree instance
     * @param <code>TreeNode</code> current node
     * @return <code>Object</code> optional return value to pass back along the path
     * @exception <code>VisitException</code> if a problem ocurred during visit
     */
    public Object visitDown( Tree tree, TreeNode node ) {

	// mark node as seen and visited
	if( ! haveSeen( node ) ) {
	    StringBuffer buf = new StringBuffer( 50 );

	    buf.append( getID( node ) );
	    
	    TreeNode[] kids = node.getActiveChildren( eval, time );
	    for( int k = 0; k < kids.length; k++ ) 
		buf.append( "\t" + getID( kids[k] ) );

	    printData( tree, buf.toString() );

	}
	 	
	return( null );
    }


    /**
     * Print visitor's header info at start of file.
     *
     * @param <code>Tree</code> current tree instance
     */
    protected void printHeader( Tree tree ) {

	// print basic info
	super.printHeader( tree );
	// add current timestamp if running with periods
	if( periods > 1 )
	    ps.println( "# timestamp=" + getTime() );
    }

    /**
     * Rotate export files if running over period intervals.
     *
     * @exception <code>VisitException</code> if processing fails
     */
    protected void process( Tree tree ) throws VisitException {

	// only need to do this of we've got multiple periods
	if( periods <= 1 )
	    return;

	// increment the period count
	long p = period++;

	// close the current export file
	close();

	// rename export file with period
	File fcur = new File( getFilename() );
	if( ! fcur.renameTo( new File( getFilename() + "." + p ) ) )
	    throw new VisitException( "Can't rename export file." );

	// don't need to do anything more unless we're going again
	if( period > periods )
	    return;

	// open a new file and print header 
	if( open() )
	    printHeader( tree );
	else
	    throw new VisitException( "Can't open export file." );
    }


    /**
     * Return the node's ID.
     * If showIP is not true, then the IP address will be mangled first.
     *
     * @param <code>TreeNode</code> current node
     * @return <code>String</code> node's ID
     */
    protected String getID( TreeNode node ) {

	String id = node.getIP();

	/* mangle IP unless requested otherwise */
	if( ! showIP ) {
	    // check if node already has an ID
	    Integer newID = (Integer)node.data.get( "id" );
	    if( newID == null ) {
		// create a new ID and store it in the node
		newID = new Integer( ++nextID );
		node.data.put( "id", newID );
	    }
	    id = newID.toString();
	}

	return( id );
    }
}
